#-------------------------------------------------------------------
#  JsonPathElement.py
#
#  The JsonPathElement class.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the JsonPathElement class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from typing import Optional
from .JsonPathParseException import JsonPathParseException
from .JsonPathResolveException import JsonPathResolveException
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonPathElement:
  '''A single element of path spec.

  For example, in path 'foo/bar[3]', 'foo' and 'bar[3]' are two elements.
  '''

  #-----------------------------------------------------------------
  def __init__(self, name: str, index: Optional[int]) -> None:
    '''Creates a new JsonPathElement.
    '''

    # The name of the element.
    self.name: str = name

    # If the element is a list, the index of the list item to choose.
    self.index: Optional[int] = index

  #----------------------------------------------------------------
  @staticmethod
  def parse(elementStr: str, wholePath: str) -> 'JsonPathElement':
    '''Parses a string into a JsonPathElement object.

    @param elementStr The string to be parsed.
    @param wholePath The whole path which the element is a part of.
                     To be used in error messages.

    @return A newly created JsonPathElement object.
    '''

    numLeft = elementStr.count('[')
    numRight = elementStr.count(']')

    if numLeft != numRight:
      msg = "Unbalanced brackets [] in element '%s'.  Path: %s"
      msg = msg % (elementStr, wholePath)
      raise JsonPathParseException(msg)

    if numLeft > 1:
      msg = ("Too many brackets [] in element '%s'.  Only one pair allowed.  "
             "Path: %s")
      msg = msg % (elementStr, wholePath)
      raise JsonPathParseException(msg)

    if numLeft == 1:

      indexLeft = elementStr.index('[')
      indexRight = elementStr.index(']')

      if indexRight < indexLeft:
        msg = "Brackets [] are reversed in element '%s'.  Path: %s"
        msg = msg % (elementStr, wholePath)
        raise JsonPathParseException(msg)

      name = elementStr[:indexLeft]
      indexStr = elementStr[indexLeft + 1:indexRight]

      try:
        index = int(indexStr)
      except ValueError:
        msg = ("Index '%s' could not be parsed as a string in element '%s'.  "
               "Path: %s")
        msg = msg % (indexStr, elementStr, wholePath)
        raise JsonPathParseException(msg)

      return JsonPathElement(name, index)

    else:

      return JsonPathElement(elementStr, None)

  #----------------------------------------------------------------
  def resolve(self, parentObject: object) -> object:
    '''Resolves this path against the specified object.

    @param parentObject The object to return a child of.

    @return The child object.
    '''

    child: object = getattr(parentObject, self.name)

    if self.index is not None:

      if not isinstance(child, list):
        msg = ("JsonPathElement %s expected to find a list, " +
               "but actually found a %s.  Value: %s")
        msg = msg % (self.toString(), type(child), child)
        raise JsonPathResolveException(msg)

      if len(child) <= self.index:
        msg = ("JsonPathElement %s expected to find a list of %s elements, " +
               "but the list actually contains %s.  List: %s")
        msg = msg % (self.toString(), self.index, len(child), child)
        raise JsonPathResolveException(msg)

      child = child[self.index]

    return child

  #----------------------------------------------------------------
  def toString(self) -> str:
    '''Returns a label for display.
    '''

    if self.index is None:
      return self.name

    else:
      return '%s[%s]' % (self.name, self.index)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['name', 'index']

    return StringUtil.formatRepr(self, attrs)
