#-------------------------------------------------------------------
#  JsonPath.py
#
#  The JsonPath class.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the JsonPath class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .JsonPathElement import JsonPathElement
from .JsonPathParseException import JsonPathParseException
from .JsonPathResolveException import JsonPathResolveException
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonPath:
  '''A path to a location in a JSON tree.
  '''

  #-----------------------------------------------------------------
  def __init__(self, elements: List[JsonPathElement]) -> None:
    '''Creates a new JsonPath.
    '''

    # The elements that make up this path.
    self.elements: List[JsonPathElement] = elements

  #----------------------------------------------------------------
  @staticmethod
  def parse(pathStr: str) -> 'JsonPath':
    '''Returns a JsonPath object for the specified string.

    The string is a slash-seperated series of names, each with
    an optional selector in brackets.

    Some examples:

      produced/bicycle/model
      produced/car/models[name]
      ../produced/car/models[name]
    '''

    if pathStr.startswith('/'):
      msg = ('Error parsing link path.  The path should be relative.  ' +
             'It should not start with a "/".  Path: ' + str(pathStr))
      raise JsonPathParseException(msg)

    else:

      elementStrs: List[str] = pathStr.split('/')

      elements: List[JsonPathElement] = []
      for elementStr in elementStrs:
        elements.append(JsonPathElement.parse(elementStr, pathStr))

      return JsonPath(elements)

  #----------------------------------------------------------------
  def resolve(self, parents: List[object], relativeTo: object) -> object:
    '''Returns the object that this path point to.

    @param parents The list of ancestor objects of the object that we're
                   starting to resove at.
    @param relativeTo This path will be resolved relative to this object.
    '''

    try:

      return self.resolveLevel(parents, relativeTo)

    except JsonPathResolveException as ex:

      msg = "Error resolving path:  %s" % self.toString()
      raise JsonPathResolveException(msg) from ex

  #----------------------------------------------------------------
  def resolveLevel(self, parents: List[object], relativeTo: object) -> object:
    '''Resolves a single level of this path.

    @param parents The list of ancestor objects of the object that we're
                   starting to resove at.
    @param relativeTo This path will be resolved relative to this object.
    '''

    if not self.elements:
      return relativeTo

    element: JsonPathElement = self.elements[0]
    pathRemainder: JsonPath = JsonPath(self.elements[1:])

    if element.name == '..':

      if not parents:
        msg = ("Error resolving path:  Can't go up a level because there no " +
               "parents.  Remaining path:  %s" % self.toString())
        raise JsonPathResolveException(msg)

      return pathRemainder.resolveLevel(parents[:-1], parents[-1])

    else:

      newParents = parents + [relativeTo]
      child = element.resolve(relativeTo)
      return pathRemainder.resolveLevel(newParents, child)

  #----------------------------------------------------------------
  def toString(self) -> str:
    '''Returns the path string for this path.
    '''

    return '/'.join([x.toString() for x in self.elements])

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['elements']

    return StringUtil.formatRepr(self, attrs)
