#-------------------------------------------------------------------
#  TestJsonPrimitiveType.py
#
#  The TestJsonPrimitiveType module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the JsonPrimitiveType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonException
from ai.axe.classJson.jsonTypes import JsonPrimitiveType
from ai.axe.build.unittest import AxeSimpleTestCase
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonPrimitiveType(AxeSimpleTestCase):
  '''Unit test for the JsonPrimitiveType class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testCtorValidation(self) -> None:
    '''Tests that you can't make a JsonPrimitiveType with an invalid type.
    '''

    try:
      JsonPrimitiveType(complex)
      self.fail("No exception thrown for bad primitive type.")
    except ClassJsonException as ex:
      self.assertEqual(True, 'Invalid type' in str(ex), "message")

  #-----------------------------------------------------------------
  def testChildJsonObjs(self) -> None:
    '''Test the childJsonObjs() methods.
    '''

    self.assertEqual([], JsonPrimitiveType(bool).childJsonObjs())

  #-----------------------------------------------------------------
  def testLabel(self) -> None:
    '''Test the toLabel() and str() methods.
    '''

    self.assertEqual('bool', JsonPrimitiveType(bool).toLabel())
    self.assertEqual('string', JsonPrimitiveType(str).toLabel())

    jsonType = JsonPrimitiveType(bool)

    self.assertIsNotNone(str(jsonType))

  #-----------------------------------------------------------------
  def testTypescriptLabel(self) -> None:
    '''Test the toTypescriptLabel() method.
    '''

    # Test str to string.
    expected = 'string'
    jsonType = JsonPrimitiveType(str)
    actual = jsonType.toTypescriptLabel('')
    self.assertEqual(expected, actual)

    # Test int and float to number.
    expected = 'number'
    jsonType = JsonPrimitiveType(int)
    actual = jsonType.toTypescriptLabel('')
    self.assertEqual(expected, actual)
    jsonType = JsonPrimitiveType(float)
    actual = jsonType.toTypescriptLabel('')
    self.assertEqual(expected, actual)

    # Test bool to boolean.
    expected = 'boolean'
    jsonType = JsonPrimitiveType(bool)
    actual = jsonType.toTypescriptLabel('')
    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testTypescriptDesc(self) -> None:
    '''Test the toTypescriptDesc() methods.
    '''

    expected = 'new JsonPrimitiveType("bool")'
    jsonType = JsonPrimitiveType(bool)

    self.assertEqual(expected, jsonType.toTypescriptDesc())

  #-----------------------------------------------------------------
  def testValidate(self) -> None:
    '''Test the validate() method.
    '''

    self.checkType(bool, True, 3.14)
    self.checkType(float, 3.14, 'invalid')
    self.checkType(int, 42, 'invalid')
    self.checkType(str, 'jim', 3.14)

    # Int is aliso valid for float.
    self.checkType(float, 3, 'invalid')

    # None is valid for anything.
    self.checkType(bool, None, 3.14)

  #-----------------------------------------------------------------
  def checkType(self,
                typeClass: type,
                validValue: object,
                invalidValue: object) -> None:
    '''Helper function to check the validate() method.
    '''

    typeName = typeClass.__name__

    jsonType = JsonPrimitiveType(typeClass)

    self.assertEqual(None, jsonType.validate(validValue), typeName + ' valid')
    self.assertEqual(None, jsonType.validateJson(validValue),
                     typeName + ' json valid')
    self.assertIsNotNone(jsonType.validate(invalidValue), typeName + ' invalid')

    self.assertEqual(None, jsonType.validate(validValue),
                     typeName + ' valid JSON')
    self.assertIsNotNone(jsonType.validate(invalidValue),
                         typeName + ' invalid JSON')

    self.assertEqual(validValue, jsonType.encode(validValue),
                     typeName + ' encode')
    self.assertEqual(validValue, jsonType.decode(validValue),
                     typeName + ' decode')

    self.assertEqual(validValue, jsonType.decodeLinks([], validValue),
                     typeName + ' decodeLinks')
