#-------------------------------------------------------------------
#  TestJsonObjRegistry.py
#
#  The TestJsonObjRegistry module.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the JsonObjRegistry class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ... import ClassJsonClass
from ... import ClassJsonClassDesc
from ... import ClassJsonException
from ... import ClassJsonField
from ..JsonObj import JsonObj
from ..JsonObjRegistry import JsonObjRegistry
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonObjRegistry(AxeSimpleTestCase):
  '''Unit test for the JsonObjRegistry class.
'''

  # pylint: disable=E1101

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    # Put finalization code here.  It will be run after each test.
    pass

  #-----------------------------------------------------------------
  def testRegistering(self) -> None:
    '''Test registering and un-registering classes.
    '''

    # pylint: disable=C0103

    class A:
      def __init__(self, num: int) -> None:
        self.num = num
    # Not registered yet.

    self.assertEqual(False, JsonObjRegistry.nameIsRegistered('A'), "before")

    try:
      JsonObjRegistry.getForName('A')
      self.fail("No exception for un-registered class.")
    except ClassJsonException:
      pass

    # Register the class.

    desc = ClassJsonClassDesc('A',
                              [ClassJsonField('num', int).toDesc()],
                              None,
                              False,
                              False,
                              False)
    jsonObj = JsonObj(A, desc)

    JsonObjRegistry.register(jsonObj)

    self.assertEqual(True, JsonObjRegistry.nameIsRegistered('A'), "registered")
    self.assertEqual(jsonObj, JsonObjRegistry.getForName('A'), "get registered")

    # Attempt to re-register the class.

    try:
      JsonObjRegistry.register(jsonObj)
      self.fail("No re-register exception.")

    except ClassJsonException as ex:
      self.assertEqual(True, 'already' in str(ex), 'already message')

    # Un-register the class.

    JsonObjRegistry.unregister("A")

    self.assertEqual(False,
                     JsonObjRegistry.nameIsRegistered("A"),
                     "un-registered")

    JsonObjRegistry.register(jsonObj)

    self.assertEqual(True,
                     JsonObjRegistry.nameIsRegistered('A'),
                     "re-registered")

    JsonObjRegistry.unregister("A")

  #-----------------------------------------------------------------
  def testFindAllClasses(self) -> None:
    '''Test finding all classes in a class.
    '''

    # pylint: disable=C0103

    # Classes whose usage tree looks like:
    #
    #       A    C
    #       ^    ^
    #       |    |
    #       B    |
    #       ^    |
    #       |    |
    #         ---
    #          |
    #          D

    @ClassJsonClass([ClassJsonField('num', int)])
    class A:
      def __init__(self, num) -> None:
        self.num = num

    @ClassJsonClass([ClassJsonField('num', int),
                     ClassJsonField('a', A)])
    class B:
      def __init__(self, num, a) -> None:
        self.num = num
        self.a = a

    @ClassJsonClass([ClassJsonField('num', int)])
    class C:
      def __init__(self, num) -> None:
        self.num = num

    @ClassJsonClass([ClassJsonField('b', B),
                     ClassJsonField('c', C)])
    class D:
      def __init__(self, b, c) -> None:
        self.b = b
        self.c = c

    jsonObjA = JsonObjRegistry.getForClass(A)
    jsonObjB = JsonObjRegistry.getForClass(B)
    jsonObjC = JsonObjRegistry.getForClass(C)
    jsonObjD = JsonObjRegistry.getForClass(D)

    # Check for used classes.

    expected: List[JsonObj] = []
    actual = JsonObjRegistry.findAllClasses(jsonObjC)
    self.assertEqual(expected, actual, "C")

    expected = [jsonObjA]
    actual = JsonObjRegistry.findAllClasses(jsonObjB)
    self.assertEqual(expected, actual, "B")

    expected = [jsonObjB, jsonObjC, jsonObjA]
    actual = JsonObjRegistry.findAllClasses(jsonObjD)
    self.assertEqual(expected, actual, "D")

    JsonObjRegistry.unregister("A")
    JsonObjRegistry.unregister("B")
    JsonObjRegistry.unregister("C")
    JsonObjRegistry.unregister("D")

  #-----------------------------------------------------------------
  def testSubclasses(self) -> None:
    '''Test finding sub/superclasses.
    '''

    # A tree of classes.
    #
    #                NamedThing
    #                    |
    #              --------------
    #             |              |
    #          NamedX         NamedAmount
    #             |
    #         NamedPoint
    #
    # Note that NamedX is not a @ClassJsonClass.  All the rest are.

    @ClassJsonClass([], isAbstract=True)
    class NamedThing:
      def __init__(self, name) -> None:
        self.name = name

    class NamedX(NamedThing):
      def __init__(self, name, x) -> None:
        NamedThing.__init__(self, name)
        self.x = x

    @ClassJsonClass([ClassJsonField('name', str),
                    ClassJsonField('x', int),
                    ClassJsonField('y', int)])
    class NamedPoint(NamedX):
      def __init__(self, name, x, y) -> None:
        NamedX.__init__(self, name, x)
        self.y = y

    @ClassJsonClass([ClassJsonField('name', str),
                     ClassJsonField('amount', int)])
    class NamedAmount(NamedThing):
      def __init__(self, name, amount) -> None:
        NamedThing.__init__(self, name)
        self.amount = amount

    joNamedThing = JsonObjRegistry.getForClass(NamedThing)
    joNamedPoint = JsonObjRegistry.getForClass(NamedPoint)
    joNamedAmount = JsonObjRegistry.getForClass(NamedAmount)

    # Find subclasses.

    expected = [joNamedPoint, joNamedAmount]

    actual = JsonObjRegistry.subclasses(joNamedThing)

    self.assertEqual(expected, actual, "subclasses")

    # Find superclasses.

    expected = [joNamedThing]

    actual = JsonObjRegistry.superclasses(joNamedAmount)
    self.assertEqual(expected, actual, "NamedAmount superclasses")

    actual = JsonObjRegistry.superclasses(joNamedPoint)
    self.assertEqual(expected, actual, "NamedPoint superclasses")

    JsonObjRegistry.unregister("NamedThing")
    JsonObjRegistry.unregister("NamedPoint")
    JsonObjRegistry.unregister("NamedAmount")
