#-------------------------------------------------------------------
#  TestJsonObj.py
#
#  The TestJsonObj module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the JsonObj class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import date
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonLink
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonObj(AxeSimpleTestCase):
  '''Unit test for the JsonObj class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister('TestClass')
    JsonObjRegistry.unregister('TestClassBase')
    JsonObjRegistry.unregister('TestClassFoo')
    JsonObjRegistry.unregister('TestClassA')
    JsonObjRegistry.unregister('TestClassB')
    JsonObjRegistry.unregister('TestClass2')
    JsonObjRegistry.unregister('TestClassMusician')
    JsonObjRegistry.unregister('TestClassInstrument')
    JsonObjRegistry.unregister('TestClassBand')

  #-----------------------------------------------------------------
  def testChildJsonObjs(self) -> None:
    '''Test the childJsonObjs() methods.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassFoo:
      def __init__(self, num: int) -> None:
        self.num = num

    classFooJsonObj = JsonObjRegistry.getForClass(TestClassFoo)

    self.assertEqual([classFooJsonObj], classFooJsonObj.childJsonObjs())

  #-----------------------------------------------------------------
  def testLabel(self) -> None:
    '''Test the toLabel() and str() methods.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    jsonType = JsonObjRegistry.getForClass(TestClass)

    self.assertIsNotNone(jsonType.toLabel())
    self.assertIsNotNone(str(jsonType))

  #-----------------------------------------------------------------
  def testTypescriptLabel(self) -> None:
    '''Test the toTypescriptLabel() method.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    jsonType = JsonObjRegistry.getForClass(TestClass)

    self.assertEqual('TestClass', jsonType.toTypescriptLabel(''))
    self.assertEqual('foo.TestClass', jsonType.toTypescriptLabel('foo'))

  #-----------------------------------------------------------------
  def testTypescriptLabelClasses(self) -> None:
    '''Test the toTypescriptLabel() method with a class tree..
    '''

    # Suppress unused variable warning.
    # pylint: disable=W0612

    @ClassJsonClass([], isAbstract=True)
    class TestClassBase:
      pass

    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassA(TestClassBase):
      def __init__(self, num: int) -> None:
        self.num = num

    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassB(TestClassBase):
      def __init__(self, num: int) -> None:
        self.num = num

    # pylint: enable=W0612

    expected = 'TestClassA | TestClassB'
    jsonType = JsonObjRegistry.getForClass(TestClassBase)
    actual = jsonType.toTypescriptLabel('')

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testBadClassType(self) -> None:
    '''A wrong class type should throw an exception.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    @ClassJsonClass([])
    class InvalidClass:
      def __init__(self) -> None:
        pass

    jsonType = JsonObjRegistry.getForClass(TestClass)

    errorMsg = jsonType.validate(InvalidClass())
    self.assertIsNotNone(errorMsg, 'not an object')

    errorMsg = jsonType.validateJson({'_class': 'InvalidClass'})
    self.assertIsNotNone(errorMsg, 'Wrong _class name.')

    errorMsg = jsonType.validateJson({'_class': 'NotRegisteredClass'})
    self.assertIsNotNone(errorMsg, 'Un-registered _class name.')

  #-----------------------------------------------------------------
  def testValidateJson(self) -> None:
    '''Test the validateJson() method.
    '''

    # pylint: disable=W0232
    # pylint: disable=W0612

    @ClassJsonClass([], isAbstract=True)
    class TestBaseClass:
      def __init__(self) -> None:
        pass

    @ClassJsonClass([], encodeOnly=True)
    class TestEncodeOnlyClass(TestBaseClass):
      def __init__(self) -> None:
        TestBaseClass.__init__(self)

    @ClassJsonClass([ClassJsonField('fooAttr', int),
                     ClassJsonField('barAttr', date)])
    class TestClass(TestBaseClass):
      def __init__(self, fooAttr: int, barAttr: date) -> None:
        TestBaseClass.__init__(self)
        self.fooAttr = fooAttr
        self.barAttr = barAttr

    jsonType = JsonObjRegistry.getForClass(TestClass)
    baseType = JsonObjRegistry.getForClass(TestBaseClass)

    # Valid objects.

    errorMsg = jsonType.validate(TestClass(3, date(2016, 11, 1)))
    self.assertIsNone(errorMsg, 'valid')

    jsonObj = {'_class': 'TestClass', 'fooAttr': 3, 'barAttr': '2016-11-1'}
    errorMsg = jsonType.validateJson(jsonObj)
    self.assertIsNone(errorMsg, 'valid')

    errorMsg = baseType.validate(TestClass(3, date(2016, 11, 1)))
    self.assertIsNone(errorMsg, 'base valid')

    jsonObj = {'_class': 'TestClass', 'fooAttr': 3, 'barAttr': '2016-11-1'}
    errorMsg = baseType.validateJson(jsonObj)
    self.assertIsNone(errorMsg, 'base valid')

    # Abstract class.

    errorMsg = baseType.validate(TestBaseClass())
    self.assertIsNotNone(errorMsg, 'abstract')

    errorMsg = baseType.validateJson({'_class': 'TestBaseClass'})
    self.assertIsNotNone(errorMsg, 'abstract')

    # Encode-only class.

    errorMsg = baseType.validateJson({'_class': 'TestEncodeOnlyClass'})
    self.assertIsNotNone(errorMsg, 'encode-only')

    # Missing _class member.

    errorMsg = jsonType.validateJson({})
    self.assertIsNotNone(errorMsg, 'missing _class attribute')

    # Missing field.

    obj = TestClass(3, date(2016, 11, 1))
    del obj.fooAttr

    errorMsg = jsonType.validate(obj)
    self.assertIsNotNone(errorMsg, 'missing field')

    errorMsg = jsonType.validateJson({'_class': 'TestClass'})
    self.assertIsNotNone(errorMsg, 'missing field')

    # Extra field.

    jsonObj = {'_class': 'TestClass',
               'fooAttr': 3,
               'barAttr': '2016-11-04',
               'fozAttr': 4,
               'bazAttr': 4}
    errorMsg = jsonType.validateJson(jsonObj)
    self.assertIsNotNone(errorMsg, 'extra field')

    # Missing and extra field.

    jsonObj = {'_class': 'TestClass',
               'barAttr': '2016-11-04',
               'fozAttr': 4,
               'bazAttr': 4}
    errorMsg = jsonType.validateJson(jsonObj)
    self.assertIsNotNone(errorMsg, 'missing and extra field')

    # Bad field type.

    invalidTypeObj = TestClass(3, 'invalid type') # type: ignore
    errorMsg = jsonType.validate(invalidTypeObj)
    self.assertIsNotNone(errorMsg, 'bad field type')

    jsonObj = {'_class': 'TestClass', 'fooAttr': 3.14, 'barAttr': 4}
    errorMsg = jsonType.validateJson(jsonObj)
    self.assertIsNotNone(errorMsg, 'bad field type')

  #-----------------------------------------------------------------
  def testSubclasses(self) -> None:
    '''Test encoding and decoding subclasses.
    '''

    # pylint: disable=W0232

    @ClassJsonClass([ClassJsonField('beginsOn', date),
                     ClassJsonField('endsOn', date)])
    class TimeRange:
      def __init__(self, beginsOn: date, endsOn: date) -> None:
        self.beginsOn = beginsOn
        self.endsOn = endsOn

    @ClassJsonClass([ClassJsonField('width', int),
                     ClassJsonField('validRange', TimeRange)])
    class Wheel:
      def __init__(self, width: int, validRange: TimeRange) -> None:
        self.width = width
        self.validRange = validRange

    @ClassJsonClass([],
                    isAbstract=True)
    class Vehicle:
      def __init__(self) -> None:
        pass

    @ClassJsonClass([ClassJsonField('wheel', Wheel)])
    class Unicycle(Vehicle):
      def __init__(self, wheel: Wheel) -> None:
        Vehicle.__init__(self)
        self.wheel = wheel

    @ClassJsonClass([ClassJsonField('wheels', [Wheel])])
    class Car(Vehicle):
      def __init__(self, wheels: List[Wheel]) -> None:
        Vehicle.__init__(self)
        self.wheels = wheels

    @ClassJsonClass([ClassJsonField('vehicles', [Vehicle])])
    class Garage:
      def __init__(self, vehicles: List[Vehicle]) -> None:
        self.vehicles = vehicles


    date1 = date(2015, 4, 1)
    date2 = date(2016, 4, 1)
    dateStr1 = '2015-04-01'
    dateStr2 = '2016-04-01'

    unicycle = Unicycle(Wheel(3, TimeRange(date1, date2)))
    wheels = [Wheel(3, TimeRange(date1, date2)),
              Wheel(3, TimeRange(date1, date2))]
    car = Car(wheels)

    garage = Garage([unicycle, car])


    unicycleJson = {
      '_class': 'Unicycle',
      'wheel': {
        '_class': 'Wheel',
        'width': 3,
        'validRange': {
          '_class': 'TimeRange',
          'beginsOn': dateStr1,
          'endsOn': dateStr2
        }
      }
    }
    wheelsJson = [
      {
        '_class': 'Wheel',
        'width': 3,
        'validRange': {
          '_class': 'TimeRange',
          'beginsOn': dateStr1,
          'endsOn': dateStr2
        }
      },
      {
        '_class': 'Wheel',
        'width': 3,
        'validRange': {
          '_class': 'TimeRange',
          'beginsOn': dateStr1,
          'endsOn': dateStr2
        }
      },
    ]

    carJson = {
      '_class': 'Car',
      'wheels': wheelsJson,
    }

    garageJson = {
      '_class': 'Garage',
      'vehicles': [
        unicycleJson,
        carJson
      ]
    }

    jsonType = JsonObjRegistry.getForClass(Garage)


    # Test Encoding.

    self.assertEqual(garageJson, jsonType.encode(garage), 'encode')

    # Test Decoding.

    actual = jsonType.decode(garageJson)

    assert isinstance(actual, Garage)
    assert isinstance(actual.vehicles[0], Unicycle)
    assert isinstance(actual.vehicles[1], Car)

    self.assertEqual(2, len(actual.vehicles), 'decode # vehicles')
    self.assertEqual(3, actual.vehicles[0].wheel.width, 'decode uni width')
    self.assertEqual(2, len(actual.vehicles[1].wheels), 'decode # wheels')

    # Clean up.

    JsonObjRegistry.unregister('TimeRange')
    JsonObjRegistry.unregister('Wheel')
    JsonObjRegistry.unregister('Vehicle')
    JsonObjRegistry.unregister('Unicycle')
    JsonObjRegistry.unregister('Car')

  #-----------------------------------------------------------------
  def testEncode(self) -> None:
    '''Test the encode() method.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    jsonType = JsonObjRegistry.getForClass(TestClass)

    self.assertIsNone(jsonType.validate(None), 'validate null')
    self.assertEqual(None, jsonType.encode(None), 'encode null')

    obj = TestClass(3)
    jsonObj = {'_class': 'TestClass', 'fooAttr': 3}

    self.assertIsNone(jsonType.validate(obj), 'validate')
    self.assertEqual(jsonObj, jsonType.encode(obj), 'encode')

  #-----------------------------------------------------------------
  def testDecode(self) -> None:
    '''Test the decode() method.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    jsonType = JsonObjRegistry.getForClass(TestClass)

    self.assertIsNone(jsonType.validateJson(None), 'validate null')
    self.assertEqual(None, jsonType.decode(None), 'decode null')

    obj = TestClass(3)
    jsonObj = {'_class': 'TestClass', 'fooAttr': 3}

    self.assertIsNone(jsonType.validateJson(jsonObj), 'validate')

    decodedObj = jsonType.decode(jsonObj)
    assert isinstance(decodedObj, TestClass)

    self.assertEqual(obj.fooAttr, decodedObj.fooAttr, 'decode')

  #-----------------------------------------------------------------
  def testDecodeLinks(self) -> None:
    '''Test the decodeLinks() method.
    '''

    # A band is a list of musicians and instruments.
    # Every musician has an instrument.

    @ClassJsonClass([ClassJsonField('id', int)])
    class TestClassInstrument:
      def __init__(self, id: int) -> None:
        self.id = id

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('instrument',
                                    ClassJsonLink(TestClassInstrument,
                                                  '../instruments',
                                                  ['id']))])
    class TestClassMusician:
      def __init__(self, id: int, instrument: TestClassInstrument) -> None:
        self.id = id
        self.instrument = instrument

    @ClassJsonClass([ClassJsonField('musicians', [TestClassMusician]),
                     ClassJsonField('instruments', [TestClassInstrument])])
    class TestClassBand:
      def __init__(self,
                   musicians: List[TestClassMusician],
                   instruments: List[TestClassInstrument]) -> None:
        self.musicians = musicians
        self.instruments = instruments

    instruments = [
      TestClassInstrument(20),
      TestClassInstrument(21),
      TestClassInstrument(22),
      ]

    musicians = [
      TestClassMusician(10, instruments[2]),
      TestClassMusician(11, instruments[0]),
      ]

    band = TestClassBand(musicians, instruments)

    # Link are unresolved, so instead of pointers to objects,
    # they are the object IDs.

    for musician in musicians:
      setattr(musician, 'instrument', [musician.instrument.id])

    jsonType = JsonObjRegistry.getForClass(TestClassMusician)
    parents: List[object] = [band]

    self.assertEqual(None, jsonType.decodeLinks(parents, None), 'null')

    jsonType.decodeLinks(parents, musicians[0])

    self.assertEqual(instruments[2], musicians[0].instrument, 'musician 0')
