#-------------------------------------------------------------------
#  TestJsonMap.py
#
#  The TestJsonMap module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the JsonMap class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonLink
from ai.axe.classJson.jsonTypes import JsonMap
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.classJson.jsonTypes import JsonPrimitiveType
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import Dict
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonMap(AxeSimpleTestCase):
  '''Unit test for the JsonMap class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister('TestClass')
    JsonObjRegistry.unregister('TestClassFoo')
    JsonObjRegistry.unregister('TestClassBar')
    JsonObjRegistry.unregister('TestClass2')
    JsonObjRegistry.unregister('TestClassMusician')
    JsonObjRegistry.unregister('TestClassInstrument')
    JsonObjRegistry.unregister('TestClassBand')

  #-----------------------------------------------------------------
  def testChildJsonObjs(self) -> None:
    '''Test the childJsonObjs() methods.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassFoo:
      def __init__(self, num: int) -> None:
        self.num = num

    @ClassJsonClass([ClassJsonField('foo', TestClassFoo)])
    class TestClassBar:
      def __init__(self, foo: TestClassFoo) -> None:
        self.foo = foo

    classFoo = JsonObjRegistry.getForClass(TestClassFoo)
    classBar = JsonObjRegistry.getForClass(TestClassBar)

    jsonType = JsonMap(classBar, JsonPrimitiveType(int))
    self.assertEqual([classBar], jsonType.childJsonObjs())
    jsonType = JsonMap(classFoo, classBar)
    self.assertEqual([classFoo, classBar], jsonType.childJsonObjs())

  #-----------------------------------------------------------------
  def testLabel(self) -> None:
    '''Test the toLabel() and str() methods.
    '''

    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(bool))

    self.assertIsNotNone(jsonType.toLabel())
    self.assertIsNotNone(str(jsonType))

  #-----------------------------------------------------------------
  def testTypescriptLabel(self) -> None:
    '''Test the toTypescriptLabel() method.
    '''

    expected = 'Map<number, boolean>'
    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(bool))
    actual = jsonType.toTypescriptLabel('')

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testTypescriptDesc(self) -> None:
    '''Test the toTypescriptDesc() methods.
    '''

    expected = ('new JsonMap(new JsonPrimitiveType("int"), ' +
                'new JsonPrimitiveType("bool"))')
    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(bool))

    self.assertEqual(expected, jsonType.toTypescriptDesc())

  #-----------------------------------------------------------------
  def testBadDictType(self) -> None:
    '''A wrong dict type should throw an exception.
    '''

    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(bool))

    errorMsg = jsonType.validate('bad value')
    self.assertIsNotNone(errorMsg)

    errorMsg = jsonType.validateJson('bad value')
    self.assertIsNotNone(errorMsg)

  #-----------------------------------------------------------------
  def testBadDictItemType(self) -> None:
    '''A wrong dict item type should throw an exception.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('foo', int)])
    class TestClass:
      def __init__(self, foo: int) -> None:
        self.foo = foo

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('foo', int)])
    class TestClass2:
      def __init__(self, foo: int) -> None:
        self.foo = foo

    jsonType = JsonMap(JsonObjRegistry.getForClass(TestClass),
                       JsonObjRegistry.getForClass(TestClass2))

    theDict = {
      TestClass(1): TestClass2(12),
      TestClass(2): TestClass2(13),
    }

    jsonTestClass1 = {'_class': 'TestClass', 'foo': 1}
    jsonTestClass2 = {'_class': 'TestClass', 'foo': 2}
    jsonTestClass3 = {'_class': 'TestClass', 'foo': 3}
    jsonTestClass12 = {'_class': 'TestClass2', 'foo': 12}
    jsonTestClass13 = {'_class': 'TestClass2', 'foo': 13}
    jsonTestClass14 = {'_class': 'TestClass2', 'foo': 14}

    jsonDict = [
      [jsonTestClass1, jsonTestClass12],
      [jsonTestClass2, jsonTestClass13],
    ]

    errorMsg = jsonType.validate(theDict)
    self.assertIsNone(errorMsg)

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNone(errorMsg)

    # Test a value of the wrong type.

    theDict[TestClass(3)] = TestClass(14) # type: ignore

    errorMsg = jsonType.validate(theDict)
    self.assertIsNotNone(errorMsg)

    jsonDict.append([jsonTestClass3, jsonTestClass1])

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNotNone(errorMsg)

    # Test a key of the wrong type.

    theDict.clear()
    theDict[TestClass(1)] = TestClass2(12)
    theDict[TestClass(2)] = TestClass2(13)
    theDict[TestClass2(3)] = TestClass2(14) # type: ignore

    errorMsg = jsonType.validate(theDict)
    self.assertIsNotNone(errorMsg)

    jsonDict[-1] = [jsonTestClass12, jsonTestClass14]

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNotNone(errorMsg)

  #-----------------------------------------------------------------
  def testBadDictItemTypeJson(self) -> None:
    '''A wrong dict item type should throw an exception.
    '''

    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(float))

    jsonDict = [[1, 1.1], [2, 2.2]]

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNone(errorMsg)

    # Test wrong length item sub-list.
    jsonDict = [[1, 1.1, 'wrong'], [2, 2.2]]  # type: ignore

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNotNone(errorMsg)

    # Test a value of the wrong type.
    jsonDict = [[1, 1.1], [2, 'bad type']]  # type: ignore

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNotNone(errorMsg)

    # Test a key of the wrong type.
    jsonDict = [[1, 1.1], ['bad type', 2.22]]  # type: ignore

    errorMsg = jsonType.validateJson(jsonDict)
    self.assertIsNotNone(errorMsg)

  #-----------------------------------------------------------------
  def testEncode(self) -> None:
    '''Test the encode() method.
    '''

    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(float))

    # Test null.

    self.assertIsNone(jsonType.validate(None), 'validate null')
    self.assertEqual(None, jsonType.encode(None), 'null')

    # Test a map.

    jsonDict = [[1, 1.1], [2, 2.2]]

    objDict = OrderedDict(((1, 1.1), (2, 2.2)))

    self.assertIsNone(jsonType.validate(objDict), 'validate')

    actual = jsonType.encode(objDict)
    self.assertEqual(jsonDict, actual, 'encode')

  #-----------------------------------------------------------------
  def testDecode(self) -> None:
    '''Test the decode() method.
    '''

    jsonType = JsonMap(JsonPrimitiveType(int), JsonPrimitiveType(float))

    # Test null.

    self.assertIsNone(jsonType.validateJson(None), 'validate null')
    self.assertEqual(None, jsonType.decode(None), 'null')

    # Test a map.

    jsonDict = [[1, 1.1], [2, 2.2]]

    objDict = OrderedDict(((1, 1.1), (2, 2.2)))

    self.assertIsNone(jsonType.validateJson(jsonDict), 'validate')

    actual = jsonType.decode(jsonDict)
    self.assertEqual(objDict, actual, 'decode')

  #-----------------------------------------------------------------
  def testDecodeLinks(self) -> None:
    '''Test the decodeLinks() method.
    '''

    # A band is a list of musicians and instruments.
    # Every musician has an instrument.

    @ClassJsonClass([ClassJsonField('id', int)])
    class TestClassInstrument:
      def __init__(self, id: int) -> None:
        self.id = id

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('instrument',
                                    ClassJsonLink(TestClassInstrument,
                                                  '../instruments',
                                                  ['id']))])
    class TestClassMusician:
      def __init__(self, id: int, instrument: TestClassInstrument) -> None:
        self.id = id
        self.instrument = instrument

    @ClassJsonClass([ClassJsonField('musicians', {str: TestClassMusician}),
                     ClassJsonField('instruments', [TestClassInstrument])])
    class TestClassBand:
      def __init__(self,
                   musicians: Dict[str, TestClassMusician],
                   instruments: List[TestClassInstrument]) -> None:
        self.musicians = musicians
        self.instruments = instruments

    instruments = [
      TestClassInstrument(20),
      TestClassInstrument(21),
      TestClassInstrument(22),
      ]

    musicians = {
      'Jane': TestClassMusician(10, instruments[2]),
      'Fred': TestClassMusician(11, instruments[0]),
      }

    band = TestClassBand(musicians, instruments)

    # Link are unresolved, so instead of pointers to objects,
    # they are the object IDs.

    for musician in musicians.values():
      setattr(musician, 'instrument', [musician.instrument.id])

    jsonType = JsonMap(JsonPrimitiveType(str),
                       JsonObjRegistry.getForClass(TestClassMusician))
    parents: List[object] = [band]

    self.assertEqual(None, jsonType.decodeLinks(parents, None), 'null')

    jsonType.decodeLinks(parents, musicians)

    self.assertEqual(instruments[2], musicians['Jane'].instrument, 'Jane')
    self.assertEqual(instruments[0], musicians['Fred'].instrument, 'Fred')
