#-------------------------------------------------------------------
#  TestJsonList.py
#
#  The TestJsonList module.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Unit test for the JsonList class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonLink
from ai.axe.classJson.jsonTypes import JsonList
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.classJson.jsonTypes import JsonPrimitiveType
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonList(AxeSimpleTestCase):
  '''Unit test for the JsonList class.
'''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister('TestClass')
    JsonObjRegistry.unregister('TestClassA')
    JsonObjRegistry.unregister('TestClassB')
    JsonObjRegistry.unregister('TestClass2')
    JsonObjRegistry.unregister('TestClassBase')
    JsonObjRegistry.unregister('TestClassFoo')
    JsonObjRegistry.unregister('TestClassMusician')
    JsonObjRegistry.unregister('TestClassInstrument')
    JsonObjRegistry.unregister('TestClassBand')

  #-----------------------------------------------------------------
  def testChildJsonObjs(self) -> None:
    '''Test the childJsonObjs() methods.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassFoo:
      def __init__(self, num: int) -> None:
        self.num = num

    classFoo = JsonObjRegistry.getForClass(TestClassFoo)

    self.assertEqual([classFoo], JsonList(classFoo).childJsonObjs())

  #-----------------------------------------------------------------
  def testLabel(self) -> None:
    '''Test the toLabel() and str() methods.
    '''

    jsonType = JsonList(JsonPrimitiveType(int))

    self.assertIsNotNone(jsonType.toLabel())
    self.assertIsNotNone(str(jsonType))

  #-----------------------------------------------------------------
  def testTypescriptLabel(self) -> None:
    '''Test the toTypescriptLabel() method.
    '''

    expected = 'Array<number>'
    jsonType = JsonList(JsonPrimitiveType(int))
    actual = jsonType.toTypescriptLabel('')

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testTypescriptLabelClasses(self) -> None:
    '''Test the toTypescriptLabel() method with a class tree..
    '''

    # Suppress unused variable warning.
    # pylint: disable=W0612

    @ClassJsonClass([], isAbstract=True)
    class TestClassBase:
      pass

    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassA(TestClassBase):
      def __init__(self, num: int) -> None:
        self.num = num

    @ClassJsonClass([ClassJsonField('num', int)])
    class TestClassB(TestClassBase):
      def __init__(self, num: int) -> None:
        self.num = num

    # pylint: enable=W0612

    expected = 'Array<TestClassA | TestClassB>'
    jsonType = JsonList(JsonObjRegistry.getForClass(TestClassBase))
    actual = jsonType.toTypescriptLabel('')

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testTypescriptDesc(self) -> None:
    '''Test the toTypescriptDesc() methods.
    '''

    expected = 'new JsonList(new JsonPrimitiveType("int"))'
    jsonType = JsonList(JsonPrimitiveType(int))

    self.assertEqual(expected, jsonType.toTypescriptDesc())

  #-----------------------------------------------------------------
  def testBadListType(self) -> None:
    '''A wrong list type should throw an exception.
    '''

    jsonType = JsonList(JsonPrimitiveType(int))

    errorMsg = jsonType.validate('bad value')
    self.assertIsNotNone(errorMsg)

    errorMsg = jsonType.validateJson('bad value')
    self.assertIsNotNone(errorMsg)

  #-----------------------------------------------------------------
  def testBadListItemType(self) -> None:
    '''A wrong list item type should throw an exception.
    '''

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('foo', int)])
    class TestClass:
      def __init__(self, foo: int) -> None:
        self.foo = foo

    # pylint: disable=W0232
    @ClassJsonClass([ClassJsonField('foo', int)])
    class TestClass2:
      def __init__(self, foo: int) -> None:
        self.foo = foo

    jsonType = JsonList(JsonObjRegistry.getForClass(TestClass))

    jsonType.validate([TestClass(1), TestClass2(2), 'bad value 2'])

    errorMsg = jsonType.validate([TestClass(1), TestClass2(2), 'bad value 2'])
    self.assertIsNotNone(errorMsg)

    jsonTestClass = {'_class': 'TestClass', 'foo': 1}
    jsonTestClass2 = {'_class': 'TestClass2', 'foo': 2}

    jsonList = [jsonTestClass, jsonTestClass2, 'bad value 2']
    errorMsg = jsonType.validateJson(jsonList)

    self.assertIsNotNone(errorMsg)

  #-----------------------------------------------------------------
  def testValidateJson(self) -> None:
    '''Test the validateJson() method.
    '''

    jsonType = JsonList(JsonPrimitiveType(int))

    self.assertIsNone(jsonType.validate([1, 2, 3]), 'list')
    self.assertIsNone(jsonType.validateJson([1, 2, 3]), 'list')

    self.assertIsNone(jsonType.validate(None), 'null')
    self.assertIsNone(jsonType.validateJson(None), 'null')

  #-----------------------------------------------------------------
  def testEncode(self) -> None:
    '''Test the encode() method.
    '''

    jsonType = JsonList(JsonPrimitiveType(int))

    self.assertEqual([3], jsonType.encode([3]), 'list')
    self.assertEqual(None, jsonType.encode(None), 'null')

  #-----------------------------------------------------------------
  def testDecode(self) -> None:
    '''Test the decode() method.
    '''

    jsonType = JsonList(JsonPrimitiveType(int))

    self.assertEqual([3], jsonType.decode([3]), 'list')
    self.assertEqual(None, jsonType.decode(None), 'null')

  #-----------------------------------------------------------------
  def testDecodeLinks(self) -> None:
    '''Test the decodeLinks() method.
    '''

    # A band is a list of musicians and instruments.
    # Every musician has an instrument.

    @ClassJsonClass([ClassJsonField('id', int)])
    class TestClassInstrument:
      def __init__(self, id: int) -> None:
        self.id = id

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('instrument',
                                    ClassJsonLink(TestClassInstrument,
                                                  '../instruments',
                                                  ['id']))])
    class TestClassMusician:
      def __init__(self, id: int, instrument: TestClassInstrument) -> None:
        self.id = id
        self.instrument = instrument

    @ClassJsonClass([ClassJsonField('musicians', [TestClassMusician]),
                     ClassJsonField('instruments', [TestClassInstrument])])
    class TestClassBand:
      def __init__(self,
                   musicians: List[TestClassMusician],
                   instruments: List[TestClassInstrument]) -> None:
        self.musicians = musicians
        self.instruments = instruments

    instruments = [
      TestClassInstrument(20),
      TestClassInstrument(21),
      TestClassInstrument(22),
      ]

    musicians = [
      TestClassMusician(10, instruments[2]),
      TestClassMusician(11, instruments[0]),
      ]

    band = TestClassBand(musicians, instruments)

    # Link are unresolved, so instead of pointers to objects,
    # they are the object IDs.

    for musician in musicians:
      setattr(musician, 'instrument', [musician.instrument.id])

    jsonType = JsonList(JsonObjRegistry.getForClass(TestClassMusician))
    parents: List[object] = [band]

    self.assertEqual(None, jsonType.decodeLinks(parents, None), 'null')

    jsonType.decodeLinks(parents, musicians)

    self.assertEqual(instruments[2], musicians[0].instrument, 'musician 0')
    self.assertEqual(instruments[0], musicians[1].instrument, 'musician 1')
