#-------------------------------------------------------------------
#  TestJsonLink.py
#
#  The TestJsonLink module.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''Unit test for the JsonLink class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.classJson import ClassJsonClass
from ai.axe.classJson import ClassJsonField
from ai.axe.classJson import ClassJsonLink
from ai.axe.classJson.jsonTypes import JsonLink
from ai.axe.classJson.jsonTypes import JsonObjRegistry
from ai.axe.build.unittest import AxeSimpleTestCase
from typing import List
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
class TestJsonLink(AxeSimpleTestCase):
  '''Unit test for the JsonLink class.
  '''

  #-----------------------------------------------------------------
  def setUp(self) -> None:

    # Put initialization code here.  It will be run before each test.
    pass

  #-----------------------------------------------------------------
  def tearDown(self) -> None:

    JsonObjRegistry.unregister('TestClass')
    JsonObjRegistry.unregister('TestClassA')
    JsonObjRegistry.unregister('TestClassB')
    JsonObjRegistry.unregister('TestClassBase')
    JsonObjRegistry.unregister('TestClass2')
    JsonObjRegistry.unregister('TestClassMusician')
    JsonObjRegistry.unregister('TestClassInstrument')
    JsonObjRegistry.unregister('TestClassBand')

  #-----------------------------------------------------------------
  def testChildJsonObjs(self) -> None:
    '''Test the childJsonObjs() methods.
    '''

    jsonType = self.createType()

    self.assertEqual([], jsonType.childJsonObjs())

  #-----------------------------------------------------------------
  def testLabel(self) -> None:
    '''Test the toLabel() and str() methods.
    '''

    jsonType = self.createType()

    self.assertIsNotNone(jsonType.toLabel())
    self.assertIsNotNone(str(jsonType))

  #-----------------------------------------------------------------
  def testTypescriptLabel(self) -> None:
    '''Test the toTypescriptLabel() method.
    '''

    expected = 'TestClass'
    jsonType = self.createType()
    actual = jsonType.toTypescriptLabel('')

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testTypescriptLabelClasses(self) -> None:
    '''Test the toTypescriptLabel() method with a class tree..
    '''

    # pylint: disable=unused-variable

    @ClassJsonClass([ClassJsonField('id', int)], isAbstract=True)
    class TestClassBase:
      pass

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('num', int)])
    class TestClassA(TestClassBase):
      def __init__(self, id: int, num: int) -> None:
        self.id = id
        self.num = num

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('num', int)])
    class TestClassB(TestClassBase):
      def __init__(self, id: int, num: int) -> None:
        self.id = id
        self.num = num

    # pylint: enable=unused-variable

    expected = 'TestClassA | TestClassB'
    jsonType = JsonLink(TestClassBase, 'ignored', ['id'])
    actual = jsonType.toTypescriptLabel('')

    self.assertEqual(expected, actual)

  #-----------------------------------------------------------------
  def testTypescriptDesc(self) -> None:
    '''Test the toTypescriptDesc() methods.
    '''

    expected = 'new JsonLink("TestClass", "a/b/c", ["id"])'
    jsonType = self.createType()

    self.assertEqual(expected, jsonType.toTypescriptDesc())

  #-----------------------------------------------------------------
  def testValidateJson(self) -> None:
    '''Test the validateJson() method.
    '''

    testClass, jsonType = self.createClassType()

    testObj = testClass(42, 43)

    self.assertIsNone(jsonType.validate(testObj), 'object')
    self.assertIsNone(jsonType.validateJson([42]), 'int')

    self.assertIsNone(jsonType.validate(None), 'null')
    self.assertIsNone(jsonType.validateJson(None), 'null')

  #-----------------------------------------------------------------
  def testClassHasNoId(self) -> None:
    '''A class that has no ID field.
    '''

    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    try:

      JsonLink(TestClass, 'ignored', ['id'])
      self.fail("No exception for class with no ID field.")

    except ValueError as ex:

      self.assertTrue('does not have a member' in str(ex), 'correct exception')

  #-----------------------------------------------------------------
  def testObjectHasNoId(self) -> None:
    '''An object that has no ID field.
    '''

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, id: int, fooAttr: int) -> None:
        self.id = id
        self.fooAttr = fooAttr

    jsonType = JsonLink(TestClass, 'ignored', ['id'])

    testObj = TestClass(42, 43)

    # Remove this object's 'id' member.
    delattr(testObj, 'id')

    self.assertIsNotNone(jsonType.validate(testObj), 'object with no ID')


  #-----------------------------------------------------------------
  def testWrongClass(self) -> None:
    '''An object that is the wrong class.
    '''

    testClass, jsonType = self.createClassType()

    # Should be a TestClass, but is actually an int.
    wrongClassObj: object = 43
    self.assertIsNotNone(jsonType.validate(wrongClassObj), 'wrong class')


    # Should be an int, but is actually a TestClass.
    wrongClassObj = testClass(42, 43)
    self.assertIsNotNone(jsonType.validateJson(wrongClassObj), 'wrong class')

  #-----------------------------------------------------------------
  def testEncode(self) -> None:
    '''Test the encode() method.
    '''

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, id: int, fooAttr: int) -> None:
        self.id = id
        self.fooAttr = fooAttr

    jsonType = JsonLink(TestClass, 'ignored', ['id'])

    testObj = TestClass(42, 43)

    self.assertEqual([42], jsonType.encode(testObj), 'encode')

  #-----------------------------------------------------------------
  def testDecode(self) -> None:
    '''Test the decode() method.
    '''

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, id: int, fooAttr: int) -> None:
        self.id = id
        self.fooAttr = fooAttr

    jsonType = JsonLink(TestClass, 'ignored', ['id'])

    self.assertEqual([3], jsonType.decode([3]), 'decode')
    self.assertEqual(None, jsonType.decode(None), 'decode null')

  #-----------------------------------------------------------------
  def testDecodeLinks(self) -> None:
    '''Test the decodeLinks() method.
    '''

    # A band is a list of musicians and instruments.
    # Every musician has an instrument.

    @ClassJsonClass([ClassJsonField('id', int)])
    class TestClassInstrument:
      def __init__(self, id: int) -> None:
        self.id = id

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('instrument',
                                    ClassJsonLink(TestClassInstrument,
                                             '../instruments',
                                             ['id']))])
    class TestClassMusician:
      def __init__(self, id: int, instrument: TestClassInstrument) -> None:
        self.id = id
        self.instrument = instrument

    @ClassJsonClass([ClassJsonField('musicians', [TestClassMusician]),
                     ClassJsonField('instruments', [TestClassInstrument])])
    class TestClassBand:
      def __init__(self,
                   musicians: List[TestClassMusician],
                   instruments: List[TestClassInstrument]) -> None:
        self.musicians = musicians
        self.instruments = instruments

    instruments = [
      TestClassInstrument(20),
      TestClassInstrument(21),
      TestClassInstrument(22),
      ]

    musicians = [
      TestClassMusician(10, instruments[2]),
      TestClassMusician(11, instruments[0]),
      ]

    band = TestClassBand(musicians, instruments)

    # Link are unresolved, so instead of pointers to objects,
    # they are the object IDs.

    for musician in musicians:
      setattr(musician, 'instrument', [musician.instrument.id])

    jsonType = JsonLink(TestClassInstrument, '../instruments', ['id'])
    parents: List[object] = [band, musicians[0]]

    self.assertEqual(None, jsonType.decodeLinks(parents, None), 'null')

    linkedObj = jsonType.decodeLinks(parents, musicians[0].instrument)

    self.assertEqual(instruments[2], linkedObj, 'musician 0')


    # Invalid state: link target is not a list:

    jsonType2 = JsonLink(TestClassInstrument, '..', ['id'])

    try:
      jsonType2.decodeLinks(parents, musicians[0].instrument)
      self.fail("No exception for non-list link target.")
    except TypeError:
      pass


    # Invalid state: link target item classes.

    @ClassJsonClass([ClassJsonField('fooAttr', int)])
    class TestClassFoo:
      def __init__(self, fooAttr: int) -> None:
        self.fooAttr = fooAttr

    jsonType3 = JsonLink(TestClassFoo, '../instruments', ['fooAttr'])

    try:
      jsonType3.decodeLinks(parents, musicians[0].instrument)
      self.fail("No exception for wrong list item type.")
    except TypeError:
      pass


    # Invalid state: link ID not found.

    badMusician = TestClassMusician(12, instruments[0])
    setattr(badMusician, 'instrument', -1)

    try:
      jsonType.decodeLinks(parents, badMusician.instrument)
      self.fail("No exception for non-list link target.")
    except TypeError:
      pass

  #-----------------------------------------------------------------
  def createType(self) -> JsonLink:
    '''Private helper metod to create a type.
    '''

    return self.createClassType()[1]

  #-----------------------------------------------------------------
  def createClassType(self) -> Tuple[type, JsonLink]:
    '''Private helper metod to create a type.
    '''

    @ClassJsonClass([ClassJsonField('id', int),
                     ClassJsonField('fooAttr', int)])
    class TestClass:
      def __init__(self, id: int, fooAttr: int) -> None:
        self.id = id
        self.fooAttr = fooAttr

    jsonType = JsonLink(TestClass, 'a/b/c', ['id'])

    return TestClass, jsonType
