#-------------------------------------------------------------------
#  JsonTypeCreation.py
#
#  The JsonTypeCreation module.
#
#  Copyright 2013 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Factory function for JsonType subclasses.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ...classJson.ClassJsonException import ClassJsonException
from ...classJson.ClassJsonLink import ClassJsonLink
from .JsonAny import JsonAny
from .JsonLink import JsonLink
from .JsonMap import JsonMap
from .JsonDate import JsonDate
from .JsonDateTime import JsonDateTime
from .JsonDuration import JsonDuration
from .JsonList import JsonList
from .JsonObjRegistry import JsonObjRegistry
from .JsonType import JsonType
from .JsonPrimitiveType import JsonPrimitiveType
from datetime import date
from datetime import datetime
from datetime import timedelta
from typing import Dict
from typing import Type
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
# Map of types to JsonType constructors.
registeredTypes: Dict[type, Type[JsonType]] = {
  timedelta: JsonDuration,
  date: JsonDate,
  datetime: JsonDateTime,
  }

#-------------------------------------------------------------------
def registerType(theType: type, theJsonType: Type[JsonType]) -> None:
  '''Adds a type to those available to ClassJson.

  @param theType A type, such as datetime, or a class.
  @param theJsonType A JsonType subclass.
  '''

  registeredTypes[theType] = theJsonType

#-------------------------------------------------------------------
def create(typeSpec: object) -> JsonType:
  '''Returns a JsonType object for the specified type.

  @param typeSpec One of:  a @ClassJsonClass-decorated class,
                  a dictionary or list, or a Python type object:
                  bool, int, float, str, date, datetime, timedelta.

  @return A JsonType subclass object.
  '''

  if isinstance(typeSpec, dict):
    theDict = typeSpec

    if len(theDict) != 1:
      msg = "Error:  type was specified as a dictionary.  "
      msg += "It should have a single key/value item, but it actually has "
      msg += str(len(theDict)) + " items.  Type spec:" + str(theDict)
      raise ClassJsonException(msg)

    keyType = next(iter(theDict.keys()))
    valueType = theDict[keyType]

    keyType = create(keyType)
    valueType = create(valueType)

    return JsonMap(keyType, valueType)

  elif isinstance(typeSpec, list):
    theList = typeSpec

    if len(theList) != 1:
      msg = "Error:  type was specified as a list.  "
      msg += "It should have a single item, but it actually has "
      msg += str(len(theList)) + " items.  Type spec:" + str(theList)
      raise ClassJsonException(msg)

    itemType = theList[0]

    return JsonList(create(itemType))

  elif typeSpec == 'any':
    return JsonAny()

  elif isinstance(typeSpec, ClassJsonLink):
    return JsonLink(typeSpec.objClass, typeSpec.objPath, typeSpec.idFields)

  elif isinstance(typeSpec, type):

    if JsonObjRegistry.classIsRegistered(typeSpec):
      return JsonObjRegistry.getForClass(typeSpec)

    elif typeSpec in registeredTypes:
      clazz: Type[JsonType] = registeredTypes[typeSpec]
      return clazz()

    elif typeSpec in JsonPrimitiveType.allowedClasses:
      return JsonPrimitiveType(typeSpec)

    else:
      msg = 'Invalid ClassJson type: %s\n'
      msg += 'Perhaps you forgot to put a @ClassJsonClass decoration on '
      msg += 'this class.'
      msg = msg % typeSpec
      raise ClassJsonException(msg)

  else:

    msg = "Error.  Specified field with unknown type: " + str(typeSpec)
    raise ClassJsonException(msg)
