#-------------------------------------------------------------------
#  JsonPrimitiveType.py
#
#  The JsonPrimitiveType class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonPrimitiveType class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from ...classJson.ClassJsonException import ClassJsonException
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
from typing import Tuple
from typing import Union
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonPrimitiveType(JsonType):
  '''Marks a class attribute as an object of a primitive type.

  Types are: bool, float, int, string.
  '''

  #-----------------------------------------------------------------
  # The classes that are allowed to be used with JsonPrimitiveType
  allowedClasses = [bool, float, int, str]

  #-----------------------------------------------------------------
  def __init__(self, typeClass: type) -> None:
    '''Creates a new JsonPrimitiveType.
    '''

    # The type that this JsonPrimitiveType object will convert.
    self.typeClass: type = typeClass

    # The python classes that will be accepted as valid by this object.
    self.validClasses: Union[Tuple[type], Tuple[type, type]] = (typeClass,)

    JsonType.__init__(self)

    # int is also a valid float type.
    if self.typeClass == float:
      self.validClasses = (float, int)

    # Verify that the type is one of the supported types.

    allowedClasses = JsonPrimitiveType.allowedClasses
    if typeClass not in allowedClasses:

      typeNames = ', '.join([c.__name__ for c in allowedClasses])

      msg = "Invalid type.  Allowed types:  %s.  Supplied type: %s"
      msg = msg % (typeNames, typeClass)

      raise ClassJsonException(msg)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    if not isinstance(value, self.validClasses):
      msg = ('is of type ' + self.typeClass.__name__ +
             ' but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None
  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    return self.validate(value)

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    return value

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    return value

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    # Suppress argument not used warning.
    # pylint: disable=self-assigning-variable
    parents = parents

    return value

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return []

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    if self.typeClass == str:
      return 'string'
    else:
      return self.typeClass.__name__

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    labels = {
      str: 'string',
      bool: 'boolean',
      float: 'number',
      int: 'number',
      }

    assert self.typeClass in labels, self.typeClass

    return labels[self.typeClass]

  #-----------------------------------------------------------------
  def toTypescriptDesc(self) -> str:
    '''Returns typescript code to create a JsonType object for this type.

    @return A typescript string.
    '''

    return 'new JsonPrimitiveType("%s")' % self.toLabel()

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['typeClass']

    return StringUtil.formatRepr(self, attrs)
