#-------------------------------------------------------------------
#  JsonObjRegistry.py
#
#  The JsonObjRegistry class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the JsonObjRegistry class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from ..ClassJsonException import ClassJsonException
from .JsonObj import JsonObj
from typing import cast
from typing import Dict
from typing import List
import inspect
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonObjRegistry:
  '''A repository for @ClassJsonClass-annotated classes.

  A class is registered under its JSON name, that is, the name
  used in the '_class' attribute in JSON.
  '''

  #-----------------------------------------------------------------
  # All classes that have been registered, keyed by class JSON name.
  registeredNames: Dict[str, JsonObj] = OrderedDict()

  #-----------------------------------------------------------------
  # All classes that have been registered, keyed by class.
  registeredClasses: Dict[type, JsonObj] = OrderedDict()

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new JsonObjRegistry.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def getForName(className: str) -> JsonObj:
    '''Returns the class registered under the specified name.

    @param className The JSON name of the class to return.

    @return A JsonObj object.
    '''

    if not JsonObjRegistry.nameIsRegistered(className):
      raise ClassJsonException('Name not registered: %s' % className)

    return JsonObjRegistry.registeredNames[className]

  #-----------------------------------------------------------------
  @staticmethod
  def nameIsRegistered(className: str) -> bool:
    '''Returns true if a class has been registered under the specified name.

    @param className The JSON name of the class to look up.

    @return True if the name is registered, otherwise False.
    '''

    return className in JsonObjRegistry.registeredNames

  #-----------------------------------------------------------------
  @staticmethod
  def getForClass(clazz: type) -> JsonObj:
    '''Returns the class registered under the specified name.

    @param clazz The Python class to return the registered JsonObj for.

    @return A JsonObj object.
    '''

    if not JsonObjRegistry.classIsRegistered(clazz):
      raise ClassJsonException('Class not registered: %s' % clazz)

    return JsonObjRegistry.registeredClasses[clazz]

  #-----------------------------------------------------------------
  @staticmethod
  def classIsRegistered(clazz: type) -> bool:
    '''Returns true if a class has been registered.

    @param className The Python class to look up.

    @return True if the class is registered, otherwise False.
    '''

    return clazz in JsonObjRegistry.registeredClasses

  #-----------------------------------------------------------------
  @staticmethod
  def register(jsonObj: JsonObj) -> None:
    '''Registers a class for use by the ClassJson system.

    @param jsonObj The class to register.
    '''

    className = jsonObj.desc.className
    clazz = jsonObj.objClass

    if className in JsonObjRegistry.registeredNames:
      existing = JsonObjRegistry.registeredNames[className]

      msg = 'Another class with the name "%s" has already been decorated '
      msg += 'with a @ClassJsonClass decorator.  '
      msg += 'Existing class: %s  New class: %s'
      msg = msg % (className, existing, clazz)

      raise ClassJsonException(msg)

    if jsonObj.objClass in JsonObjRegistry.registeredClasses:

      msg = 'Class "%s": has already been registered.' % clazz

      raise ClassJsonException(msg)

    JsonObjRegistry.registeredNames[className] = jsonObj
    JsonObjRegistry.registeredClasses[clazz] = jsonObj

  #-----------------------------------------------------------------
  @staticmethod
  def unregister(className: str) -> None:
    '''Un-registers a class previously registered for use by ClassJson.

    @param className The name of the class as it was previously registered.
    '''

    if className in JsonObjRegistry.registeredNames:
      clazz = JsonObjRegistry.registeredNames[className].objClass
      del JsonObjRegistry.registeredNames[className]
      del JsonObjRegistry.registeredClasses[clazz]

  #-----------------------------------------------------------------
  @staticmethod
  def subclasses(jsonObj: JsonObj) -> List[JsonObj]:
    '''Returns a list of all subclasses of a class that are a ClassJsonClass.
    '''

    subclasses: List[JsonObj] = []

    for candidateJsonObj in JsonObjRegistry.registeredClasses.values():

      if jsonObj in JsonObjRegistry.superclasses(candidateJsonObj):
        subclasses.append(candidateJsonObj)

    return subclasses

  #-----------------------------------------------------------------
  @staticmethod
  def superclasses(jsonObj: JsonObj) -> List[JsonObj]:
    '''Returns a list of all bases of a class that are a ClassJsonClass.
    '''

    clazz = jsonObj.objClass

    allSuperclasses = inspect.getmro(clazz)

    superclasses = list(allSuperclasses)

    # Remove this class from the list.
    assert superclasses[0] is clazz
    superclasses.pop(0)

    classJsonSupers = []

    for superclass in superclasses:

      if JsonObjRegistry.classIsRegistered(superclass):
        classJsonSupers.append(JsonObjRegistry.getForClass(superclass))

    return classJsonSupers

  #-----------------------------------------------------------------
  @staticmethod
  def findAllClasses(jsonObj: JsonObj) -> List[JsonObj]:
    '''Returns all ClassJsonClasses in the specified class or children.
    '''

    children: List[JsonObj] = []
    descendants: List[JsonObj] = []

    children.extend(JsonObjRegistry.subclasses(jsonObj))

    for classJsonField in jsonObj.desc.properties:
      jsonType = classJsonField.jsonType
      childJsonObjs = jsonType.childJsonObjs()

      for childJsonType in childJsonObjs:

        assert isinstance(childJsonType, JsonObj)
        childJsonObj = cast(JsonObj, childJsonType)

        children.append(childJsonObj)
        descendants.extend(JsonObjRegistry.findAllClasses(childJsonObj))

    return children + descendants
