#-------------------------------------------------------------------
#  JsonObj.py
#
#  The JsonObj class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonObj class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ..ClassJsonClassDesc import ClassJsonClassDesc
from ..ClassJsonFieldDesc import ClassJsonFieldDesc
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from collections import OrderedDict
from ai.axe.util import StringUtil
from typing import cast
from typing import Callable
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
import inspect
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonObj(JsonType):
  '''Marks a class attribute as an object of a ClassJson type.
  '''

  #-----------------------------------------------------------------
  def __init__(self, objClass: type, desc: ClassJsonClassDesc) -> None:
    '''Creates a new JsonObj.
    '''

    JsonType.__init__(self)

    # The class of the object to be encoded/decoded.  Must be a ClassJsonClass.
    self.objClass: type = objClass

    # The description for how to encode/decode objects of the class.
    self.desc: ClassJsonClassDesc = desc

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    if value is None:
      return None

    if not isinstance(value, self.objClass):
      msg = ('must be of type ' + self.desc.className +
             ' but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    clazz = value.__class__

    # Get the JsonObj for the concrete subclass.

    if not JsonObjRegistry.classIsRegistered(clazz):

      msg = ('is of type %s, but that class does not have ' +
             'a @ClassJsonClass decoration.') % clazz
      return JsonTypeError(msg)

    jsonObj: JsonObj = JsonObjRegistry.getForClass(clazz)

    props: List[ClassJsonFieldDesc] = jsonObj.desc.properties

    if jsonObj.desc.isAbstract:
      msg = ('cannot be encoded because it is of abstract class ' +
             jsonObj.desc.className + '.  That class is decorated with a ' +
             '@ClassJsonClass with isAbstract=True.  An abstract class ' +
             'cannot be encoded or decoded.')
      return JsonTypeError(msg)

    missingProps = self.checkObjProps(props, value)

    if missingProps:
      return JsonTypeError('has missing attributes: ' + ', '.join(missingProps))

    for prop in props:
      propMsg = prop.jsonType.validate(getattr(value, prop.name))
      if propMsg:
        propMsg.prependPath(prop.name)
        return propMsg

    return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    if value is None:
      return None

    if not isinstance(value, dict):
      raise TypeError("Expected type dict, got %s." % type(value))

    propDict = cast(Dict[str, object], value)

    if '_class' not in propDict:
      return JsonTypeError("is missing attribute '_class'")

    classNameObj = propDict['_class']
    if not isinstance(classNameObj, str):
      msg = 'Expected _class of type str, but got: %s' % type(classNameObj)
      return JsonTypeError(msg)
    className: str = classNameObj

    # Get the JsonObj for the concrete subclass.

    if not JsonObjRegistry.nameIsRegistered(className):
      msg = "has _class=%s, but that class doesn't have a @ClassJsonClass "
      msg += "decoration."
      msg = msg % className
      return JsonTypeError(msg)

    jsonObj: JsonObj = JsonObjRegistry.getForName(className)
    desc = jsonObj.desc
    props = desc.properties

    if self.objClass not in inspect.getmro(jsonObj.objClass):
      msg = "must be type %s, but the JSON has _class=%s"
      msg = msg % (self.desc.className, className)
      return JsonTypeError(msg)

    if desc.encodeOnly:
      msg = ('is an object of class ' +
             desc.className + ', but that class is decorated with a ' +
             '@ClassJsonClass with encodeOnly=True.  An encodeOnly class' +
             'cannot be decoded.')
      return JsonTypeError(msg)

    if desc.isAbstract:
      msg = ('is an object of abstract class ' +
             desc.className + '.  That class is decorated with a ' +
             '@ClassJsonClass with isAbstract=True.  An abstract class ' +
             'cannot be encoded or decoded.')
      return JsonTypeError(msg)

    extra, missing = self.checkJsonObjProps(props, propDict)
    if extra or missing:
      msg = 'has invalid properties. '

      if extra:
        msg += 'The following extra properties were found: ' + str(extra)
      if extra and missing:
        msg += '\n'
      if missing:
        msg += 'The following properties were missing: ' + str(missing)
      return JsonTypeError(msg)

    for prop in props:
      propMsg = prop.jsonType.validateJson(propDict[prop.name])
      if propMsg:
        propMsg.prependPath(prop.name)
        return propMsg

    return None

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    if value is None:
      return None

    elif not isinstance(value, self.objClass):
      msg = "Expected %s, got: %s" % (type(self.objClass), type(value))
      raise TypeError(msg)

    clazz = value.__class__

    if clazz != self.objClass:

      # We have a concrete class.  Find the concrete JsonObj.

      if not JsonObjRegistry.classIsRegistered(clazz):

        msg = ('Encoding type %s, but that class does not have ' +
               'a @ClassJsonClass decoration.') % clazz
        raise TypeError(msg)

      concreteObj: JsonObj = JsonObjRegistry.getForClass(clazz)
      return concreteObj.encodeObj(value)

    return self.encodeObj(value)

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    if value is None:
      return None

    assert isinstance(value, dict)

    return self.decodeObj(value)

  #-----------------------------------------------------------------
  def encodeObj(self, obj: object) -> Dict[str, object]:
    '''Encodes the specified object into a JSON-ready dictionary.

    @param obj A @ClassJsonClass-decorated object to be converted to JSON.

    @return A dictionary of properties ready to be converted into JSON.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    ret: Dict[str, object] = OrderedDict()

    assert isinstance(obj, self.objClass)

    # Get the JsonObj for the concrete subclass.
    clazz = obj.__class__
    jsonObj: JsonObj = JsonObjRegistry.getForClass(clazz)

    desc: ClassJsonClassDesc = jsonObj.desc

    assert not desc.isAbstract

    ret['_class'] = desc.className

    for prop in desc.properties:
      assert hasattr(obj, prop.name), 'Missing property %s' % prop.name

      value = getattr(obj, prop.name)
      value = prop.jsonType.encode(value)
      ret[prop.name] = value

    return ret

  #-----------------------------------------------------------------
  def decodeObj(self, propDict: Dict[str, object]) -> object:
    '''Decodes the specified dictionary of object properties from JSON.

    @param propDict A dictionary of object properties to decode into an object.

    @return A newly created object.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    assert '_class' in propDict

    className = propDict['_class']
    assert isinstance(className, str)
    assert JsonObjRegistry.nameIsRegistered(className)

    # Get the JsonObj for the concrete subclass.
    jsonObj = JsonObjRegistry.getForName(className)
    desc = jsonObj.desc
    props = desc.properties

    assert self.objClass in inspect.getmro(jsonObj.objClass)
    assert not desc.encodeOnly
    assert not desc.isAbstract

    extra, missing = self.checkJsonObjProps(props, propDict)
    assert not (extra or missing), (extra, missing)

    ctorFunc: Callable[..., object] = jsonObj.objClass
    if desc.ctor:
      ctorFunc = desc.ctor

    if desc.ctorArgs:

      # Pass the args into the ctor.
      kwargs = {}
      for prop in props:
        propType = prop.jsonType
        value = propDict[prop.name]
        assert not propType.validateJson(value), propType.validateJson(value)

        value = propType.decode(value)
        kwargs[prop.name] = value
      obj = ctorFunc(**kwargs)

    else:

      # Create the object, then set the arguments.
      obj = ctorFunc()
      for prop in props:
        propType = prop.jsonType
        value = propDict[prop.name]
        assert not propType.validateJson(value), propType.validateJson(value)

        value = propType.decode(value)
        setattr(obj, prop.name, value)

    return obj

  #-----------------------------------------------------------------
  def checkObjProps(self,
                    propList: List[ClassJsonFieldDesc],
                    obj: object) -> List[str]:
    '''Returns a list of missing properties.
    '''

    expectedNames = []
    for prop in propList:
      expectedNames.append(prop.name)

    missing = []
    for name in expectedNames:
      if not hasattr(obj, name):
        missing.append(name)

    return missing

  #-----------------------------------------------------------------
  def checkJsonObjProps(self,
                        propList: List[ClassJsonFieldDesc],
                        propDict: Dict[str, object]) -> Tuple[List[str],
                                                             List[str]]:
    '''Returns an ([extra], [missing]) tuple of properties.
    '''

    expectedNames: List[str] = []
    for prop in propList:
      expectedNames.append(prop.name)

    actualNames: List[str] = []
    for propName in propDict:
      actualNames.append(propName)

    missing: List[str] = []
    for name in expectedNames:
      if name not in actualNames:
        missing.append(name)

    extra: List[str] = []
    for name in actualNames:
      if name != '_class' and name not in expectedNames:
        extra.append(name)

    return extra, missing

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    if value is None:
      return None

    # Get the JsonObj for the concrete subclass.
    jsonObj = JsonObjRegistry.getForClass(value.__class__)
    desc = jsonObj.desc
    props = desc.properties

    childParents = parents + [value]

    for prop in props:
      propType = prop.jsonType
      oldValue = getattr(value, prop.name)

      newValue = propType.decodeLinks(childParents, oldValue)
      if oldValue is not newValue:
        setattr(value, prop.name, newValue)

    return value

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return [self]

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    return self.objClass.__name__.split('.')[-1]

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    # Must import in function because of cyclical include.
    # pylint: disable=import-outside-toplevel
    from .JsonObjRegistry import JsonObjRegistry


    isAbstract: bool = self.desc.isAbstract
    subclasses: List[JsonObj] = JsonObjRegistry.subclasses(self)

    assert (not isAbstract) or subclasses, self.objClass

    names: List[str] = []

    if not isAbstract:
      names.append(JsonObj.classToTypescriptLabel(self, namespace))

    for subclass in subclasses:
      names.append(JsonObj.classToTypescriptLabel(subclass, namespace))

    return ' | '.join(names)

  #-----------------------------------------------------------------
  @staticmethod
  def classToTypescriptLabel(jsonObj: 'JsonObj', namespace: str) -> str:
    '''Returns a label for a single class for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    name = jsonObj.desc.className

    if namespace:
      name = namespace + '.' + name

    return name

  #-----------------------------------------------------------------
  def toTypescriptDesc(self) -> str:
    '''Returns typescript code to create a JsonType object for this type.

    @return A typescript string.
    '''

    return 'new JsonObj("%s")' % self.desc.className

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['objClass',
             'desc']

    return StringUtil.formatRepr(self, attrs)
