#-------------------------------------------------------------------
#  JsonMap.py
#
#  The JsonMap class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonMap class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections.abc import Mapping
from collections.abc import MutableMapping
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from ai.axe.util import StringUtil
from typing import Dict
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonMap(JsonType):
  '''Marks a class attribute as a dictionary of a ClassJson type.
  '''

  #-----------------------------------------------------------------
  def __init__(self, keyType: JsonType, valueType: JsonType) -> None:
    '''Creates a new JsonMap.
    '''

    # The type of keys in this list.
    self.keyType: JsonType = keyType

    # The type of values in this list.
    self.valueType: JsonType = valueType

    JsonType.__init__(self)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    if not isinstance(value, Mapping):
      msg = ('is type dict but the value is of type ' +
             StringUtil.typeName(value) + '.  Value: ' + str(value))
      return JsonTypeError(msg)

    valueDict = value
    del value

    for key, itemValue in valueDict.items():
      keyErrorMsg = self.keyType.validate(key)
      valueErrorMsg = self.valueType.validate(itemValue)

      if keyErrorMsg:
        keyErrorMsg.prependPathKey(key)
        return keyErrorMsg
      if valueErrorMsg:
        valueErrorMsg.prependPathKey(key)
        return valueErrorMsg

    return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    if not isinstance(value, list):
      msg = ('is type dict, so the JSON should have a list, but the value is ' +
             'of type ' + StringUtil.typeName(value) + '.  Value: ' +
             str(value))
      return JsonTypeError(msg)

    valueList = value

    for i in range(0, len(valueList)):
      item = valueList[i]
      if len(item) != 2:
        msg = (self.toLabel() + " list item " + str(i) + " " +
               "should be length 2, but item is length " + str(len(item))  +
               ". Raw values: " + str(value))
        return JsonTypeError(msg)

      key, value = item
      keyErrorMsg = self.keyType.validateJson(key)
      valueErrorMsg = self.valueType.validateJson(value)

      if keyErrorMsg:
        keyErrorMsg.prependPathKey(key)
        return keyErrorMsg
      if valueErrorMsg:
        valueErrorMsg.prependPathKey(key)
        return valueErrorMsg

    return None

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, Mapping):
      raise TypeError("Expected Mapping, got: %s" % type(value))

    valueDict = value
    del value

    # The dict encoding is a list.  This allows non-string keys to be used.

    theList = []

    for key, itemValue in valueDict.items():
      encodedKey = self.keyType.encode(key)
      encodedValue = self.valueType.encode(itemValue)

      theList.append([encodedKey, encodedValue])

    return theList

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, list):
      raise TypeError("Expected list, got: %s" % type(value))

    valueList = value
    valueDict: Dict[object, object] = {}

    for i in range(0, len(valueList)):
      item = valueList[i]
      assert len(item) == 2, len(item)

      key, value = item
      valueDict[self.keyType.decode(key)] = self.valueType.decode(value)

    return valueDict

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, MutableMapping):
      raise TypeError("Expected MutableMapping, got: %s" % type(value))

    valueDict = value
    del value

    for key, valueItem in valueDict.items():
      newItem = self.valueType.decodeLinks(parents, valueItem)
      if newItem is not valueItem:
        valueDict[key] = newItem

    return valueDict

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return self.keyType.childJsonObjs() + self.valueType.childJsonObjs()

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    return '{' + self.keyType.toLabel() + ': ' + self.valueType.toLabel() + '}'

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    keyLabel = self.keyType.toTypescriptLabel(namespace)
    valueLabel = self.valueType.toTypescriptLabel(namespace)

    return 'Map<' + keyLabel + ', ' + valueLabel + '>'

  #-----------------------------------------------------------------
  def toTypescriptDesc(self) -> str:
    '''Returns typescript code to create a JsonType object for this type.

    @return A typescript string.
    '''

    keyDesc = self.keyType.toTypescriptDesc()
    valueDesc = self.valueType.toTypescriptDesc()

    return 'new JsonMap(%s, %s)' % (keyDesc, valueDesc)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['keyType', 'valueType']

    return StringUtil.formatRepr(self, attrs)
