#-------------------------------------------------------------------
#  JsonList.py
#
#  The JsonList class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonList class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from collections.abc import Iterable
from collections.abc import MutableSequence
from ai.axe.util import StringUtil
from typing import Any
from typing import List
from typing import MutableSequence as TypingMutableSequence
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonList(JsonType):
  '''Marks a class attribute as a list of a ClassJson type.
  '''

  #-----------------------------------------------------------------
  def __init__(self, itemType: JsonType) -> None:
    '''Creates a new JsonList.
    '''

    # The type of the objects in this list.
    self.itemType: JsonType = itemType

    JsonType.__init__(self)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    if not isinstance(value, Iterable):
      msg = ('is of type list but the value is of type ' +
             StringUtil.typeName(value) + '.  Value: ' + str(value))
      return JsonTypeError(msg)

    for i, item in enumerate(value):
      errorMsg = self.itemType.validate(item)

      if errorMsg:
        errorMsg.prependPathIndex(i)
        return errorMsg

    return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    if not isinstance(value, list):
      msg = ('is of type list but the value is of type ' +
             StringUtil.typeName(value) + '.  Value: ' + str(value))
      return JsonTypeError(msg)

    for i, item in enumerate(value):
      errorMsg = self.itemType.validateJson(item)

      if errorMsg:
        errorMsg.prependPathIndex(i)
        return errorMsg

    return None

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, Iterable):
      raise TypeError("Expected Iterable, got: %s" % type(value))

    valueList = value
    del value

    ret = []

    for valueItem in valueList:
      ret.append(self.itemType.encode(valueItem))

    return ret

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, list):
      raise TypeError("Expected list, got: %s" % type(value))

    valueList = value
    del value

    ret = []

    for valueItem in valueList:
      ret.append(self.itemType.decode(valueItem))

    return ret

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, MutableSequence):
      raise TypeError("Expected MutableSequence, got: %s" % type(value))

    valueList: TypingMutableSequence[Any] = value
    del value

    for i, valueItem in enumerate(valueList):
      valueList[i] = self.itemType.decodeLinks(parents, valueItem)

    return valueList

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return self.itemType.childJsonObjs()

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    return '[' + self.itemType.toLabel() + ']'


  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    return 'Array<' + self.itemType.toTypescriptLabel(namespace) + '>'

  #-----------------------------------------------------------------
  def toTypescriptDesc(self) -> str:
    '''Returns typescript code to create a JsonType object for this type.

    @return A typescript string.
    '''

    return 'new JsonList(%s)' % self.itemType.toTypescriptDesc()

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['itemType']

    return StringUtil.formatRepr(self, attrs)
