#-------------------------------------------------------------------
#  JsonLink.py
#
#  The JsonLink class.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the JsonLink class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from ..link import JsonPath
from .JsonObj import JsonObj
from .JsonObjRegistry import JsonObjRegistry
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonLink(JsonType):
  '''Attribute that is a pointer to an object somewhere else in the JSON tree.
  '''

  #-----------------------------------------------------------------
  def __init__(self, objClass: type, objPath: str, idFields: List[str]) -> None:
    '''Creates a new JsonLink.
    '''

    JsonType.__init__(self)

    # The class of the object this is a link to.  Must be a ClassJsonClass.
    self.objClass: type = objClass

    # The JsonObj for the object that this is a link to.
    self.linkedJsonObj: JsonObj = JsonObjRegistry.getForClass(self.objClass)

    # The path to find the object this is a link to.
    self.objPath: JsonPath = JsonPath.parse(objPath)

    # The member variable to use to find the correct object in the collection.
    self.idFields: List[str] = idFields

    for idField in self.idFields:
      if not self.linkedJsonObj.desc.hasProperty(idField):
        msg = ('Error creating ClassJsonLink.  ' +
               'This is a link with idFields ' + str(self.idFields) +
               'to an object of type ' +
               self.objClass.__name__  +
               ' but that type does not have a member called "' +
               idField + '".')
        raise ValueError(msg)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    if not isinstance(value, self.objClass):
      msg = ('is of type ' + self.objClass.__name__  +
             ' but the value is of type ' +
             StringUtil.typeName(value) + '.  Value: ' + str(value))
      return JsonTypeError(msg)

    for idField in self.idFields:
      if not hasattr(value, idField):
        msg = ('uses link fields ' + str(self.idFields) +
               ' but the value is missing field "' + idField + '". ' +
               'Value: ' + str(value))
        return JsonTypeError(msg)

    return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    elif not isinstance(value, list):

      msg = ('is link ID fields ' + str(self.idFields) +
             ', but has invalid type "' + StringUtil.typeName(value)+ '". ' +
             'Value: ' + str(value))
      return JsonTypeError(msg)

    for item in value:
      if not isinstance(item, (int, str)):

        msg = ('is link ID fields ' + str(self.idFields) +
               ', but has invalid type "' + StringUtil.typeName(item)+ '". ' +
               'Value: ' + str(value))
        return JsonTypeError(msg)

    return None

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    for idField in self.idFields:
      assert hasattr(value, idField)

    # Return the ID value that can be used to find the object at decode time.
    return [getattr(value, x) for x in self.idFields]

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    # Return the ID value unchanged.  We will go find the actual object later.
    return value

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    # The 'value' is an attribute of an object, so it should have
    # at least one parent, the object it belongs to.
    assert parents

    if value is None:
      return value

    linkedObj = self.objPath.resolve(parents[:-1], parents[-1])

    if not isinstance(linkedObj, list):
      msg = ('Link %s expects a list of %s objects, but instead ' +
             'found %s object: %s')
      msg = msg % (self.objPath.toString(),
                   self.objClass.__name__,
                   StringUtil.typeName(linkedObj),
                   linkedObj)
      raise TypeError(msg)

    for i, item in enumerate(linkedObj):

      if not isinstance(item, self.objClass):
        msg = ('Link %s expects a list of type %s, but item %s ' +
               'has type %s.  Value: %s')
        msg = msg % (self.objPath.toString(),
                     self.objClass.__name__,
                     i,
                     StringUtil.typeName(item),
                     item)
        raise TypeError(msg)

      itemIdValue = [getattr(item, x) for x in self.idFields]
      if itemIdValue == value:
        return item

    msg = 'Link %s failed to find an object with %s = %s.'
    msg = msg % (self.objPath.toString(), self.idFields, value)
    raise TypeError(msg)

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List['JsonType']:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return []

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    typeLabel: str = self.linkedJsonObj.toLabel()
    label = 'link(type=%s, path="%s")' % (typeLabel, self.objPath)

    return label

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    return self.linkedJsonObj.toTypescriptLabel(namespace)

  #-----------------------------------------------------------------
  def toTypescriptDesc(self) -> str:
    '''Returns typescript code to create a JsonType object for this type.

    @return A typescript string.
    '''

    className = '"' + self.linkedJsonObj.desc.className + '"'
    path = '"' + self.objPath.toString() + '"'
    idFieldsStr = '["' + '", "'.join(self.idFields) + '"]'

    return 'new JsonLink(%s, %s, %s)' % (className, path, idFieldsStr)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['objClass', 'objPath', 'idFields']

    return StringUtil.formatRepr(self, attrs)
