#-------------------------------------------------------------------
#  JsonDuration.py
#
#  The JsonDuration class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonDuration class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import timedelta
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonDuration(JsonType):
  '''Marks a class attribute as a duration.

  This is a timedelta in python and an integer number of ms in JSON.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new JsonDuration.
    '''

    JsonType.__init__(self)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if not (value is None or isinstance(value, timedelta)):
      msg = ('is of type duration (timedelta), ' +
             'but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if not (value is None or isinstance(value, int)):
      msg = ('is of type duration (int), ' +
             'but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, timedelta):
      raise TypeError("Expected timedelta, got: %s" % value)

    else:

      # Return integer milliseconds.

      secs = value.total_seconds()
      mSecs = secs * 1000
      mSecs = int(round(mSecs))

      return mSecs

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, int):
      raise TypeError("Expected int, got: %s" % value)

    else:
      return timedelta(milliseconds=value)

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    # Suppress argument not used warning.
    # pylint: disable=self-assigning-variable
    parents = parents

    return value

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return []

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    return 'duration'

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    return 'Duration'

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
