#-------------------------------------------------------------------
#  JsonDateTime.py
#
#  The JsonDateTime class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonDateTime class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import datetime
from datetime import timedelta
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonDateTime(JsonType):
  '''Marks a class attribute as a datetime.

  This is a datetime object in python and an ISO 8601 string in JSON.
  '''

  #-----------------------------------------------------------------
  # The pattern to use for formatting datetimes.
  DATETIME_PATTERN = "%Y-%m-%dT%H:%M:%S"

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new JsonDateTime.
    '''

    JsonType.__init__(self)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if not (value is None or isinstance(value, datetime)):
      msg = ('is of type datetime, ' +
             'but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    elif not isinstance(value, str):
      msg = ('is of type datetime (string), ' +
             'but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    elif not self.canBeParsed(value):
      msg = ('is of type datetime, and the value is not valid format "' +
             JsonDateTime.DATETIME_PATTERN + '.sssZ"' +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None

  #-----------------------------------------------------------------
  def canBeParsed(self, dateTimeStr: str) -> bool:
    '''Returns True if the specified string can be parsed as a datetime.
    '''

    try:
      self.datetimeStringToObject(dateTimeStr)
      return True

    except ValueError:
      return False

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, datetime):
      raise TypeError("Expected datetime, got: %s" % value)

    else:

      return self.encodeDatetime(value)

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    if value is None:
      return None
    elif not isinstance(value, str):
      raise TypeError("Expected str, got: %s" % value)
    else:
      return JsonDateTime.datetimeStringToObject(value)

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    # Suppress argument not used warning.
    # pylint: disable=self-assigning-variable
    parents = parents

    return value

  #-----------------------------------------------------------------
  @staticmethod
  def datetimeStringToObject(datetimeStr: str) -> datetime:
    '''Decodes a datetime string into a datetime object.
    '''

    # Remove the trailing time zone code.
    datetimeStr = datetimeStr.rstrip("Z")

    # strptime doesn't handle miliseconds, so split them off and parse
    # seperately.
    dtNoMillis, unusedSep, millisStr = datetimeStr.partition(".")
    dtObj = datetime.strptime(dtNoMillis, JsonDateTime.DATETIME_PATTERN)

    # Any precision beyond milliseconds is silently ignored.
    millisStr = millisStr[:3]

    milliseconds = int(millisStr, base=10) if millisStr else 0

    return dtObj + timedelta(milliseconds=milliseconds)

  #-----------------------------------------------------------------
  def encodeDatetime(self, obj: datetime) -> str:
    '''Encodes a datetime into JSON.

    The datetime encoding is a ISO date string.
    '''

    dateStr = obj.strftime(JsonDateTime.DATETIME_PATTERN)

    # This format has milliseconds, even though datetime supports microseconds.
    # This is because Javascript dates support only milliseconds.

    millis = obj.microsecond / 1000
    millis = int(millis)
    milliStr = str(millis)

    # Pad with zeros.
    while len(milliStr) < 3:
      milliStr = '0' + milliStr

    dateStr = dateStr + '.' + str(milliStr) + 'Z'

    return dateStr

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return []

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    return 'datetime'

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    return 'Date'

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
