#-------------------------------------------------------------------
#  JsonDate.py
#
#  The JsonDate class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the JsonDate class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from datetime import date
from datetime import datetime
from .JsonType import JsonType
from .JsonTypeError import JsonTypeError
from ai.axe.util import StringUtil
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class JsonDate(JsonType):
  '''Marks a class attribute as a date.

  This is a date object in python and an ISO 8601 string in JSON.
  '''

  #-----------------------------------------------------------------
  # The pattern to use for formatting dates.
  DATE_PATTERN = "%Y-%m-%d"

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new JsonDate.
    '''

    JsonType.__init__(self)

  #-----------------------------------------------------------------
  def validate(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified value can be converted to JSON.

    @param value The value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if not (value is None or isinstance(value, date)):
      msg = ('is of type date, ' +
             'but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None

  #-----------------------------------------------------------------
  def validateJson(self, value: object) -> Optional[JsonTypeError]:
    '''Checks that the specified JSON string can be converted to an object.

    @param value The JSON value to validate.

    @return None if the value is OK, or a JsonTypeError if there is a problem.
    '''

    if value is None:
      return None

    elif not isinstance(value, str):
      msg = ('is of type date (string), ' +
             'but the value is of type ' + StringUtil.typeName(value) +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    elif not self.canBeParsed(value):
      msg = ('is of type date, and the value is not valid date format "' +
             JsonDate.DATE_PATTERN + '"' +
             '.  Value: ' + str(value))
      return JsonTypeError(msg)

    else:
      return None

  #-----------------------------------------------------------------
  def canBeParsed(self, dateStr: str) -> bool:
    '''Returns True if the specified string can be parsed as a date.
    '''

    try:
      datetime.strptime(dateStr, JsonDate.DATE_PATTERN)
      return True

    except ValueError:
      return False

  #-----------------------------------------------------------------
  def encode(self, value: object) -> object:
    '''Encodes a value into JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, date):
      raise TypeError("Expected date, got: %s" % value)

    else:

      dateObj: date = value
      return dateObj.isoformat()

  #-----------------------------------------------------------------
  def decode(self, value: object) -> object:
    '''Decodes a value from a JSON-ready value.
    '''

    if value is None:
      return None

    elif not isinstance(value, str):
      raise TypeError("Expected str, got: %s" % value)

    else:
      return JsonDate.dateStringToObject(value)

  #-----------------------------------------------------------------
  def decodeLinks(self, parents: List[object], value: object) -> object:
    '''Decodes any links in a JSON-ready value.
    '''

    # Suppress argument not used warning.
    # pylint: disable=self-assigning-variable
    parents = parents

    return value

  #-----------------------------------------------------------------
  @staticmethod
  def dateStringToObject(dateStr: str) -> date:
    '''Decodes a date string into a date object.
    '''

    datetimeObj = datetime.strptime(dateStr, JsonDate.DATE_PATTERN)
    dateObj = datetimeObj.date()
    return dateObj

  #-----------------------------------------------------------------
  def childJsonObjs(self) -> List[JsonType]:
    '''Returns all JsonObj types that are children of this type.

    @return A list of JsonObj objects.
    '''

    return []

  #-----------------------------------------------------------------
  def toLabel(self) -> str:
    '''Returns a label for this type for display for a user.

    @return A string.
    '''

    return 'date'

  #-----------------------------------------------------------------
  def toTypescriptLabel(self, namespace: str) -> str:
    '''Returns a label for this type for display for a user.

    @param namespace The namespace that any types should be placed in.

    @return A typescript string.
    '''

    return 'Day'

  #----------------------------------------------------------------
  def __repr__(self):
    '''Returns a string representation of this object
    '''
    attrs = []

    return StringUtil.formatRepr(self, attrs)
