#-------------------------------------------------------------------
#  ClassJsonTypescriptWriter.py
#
#  The ClassJsonTypescriptWriter class.
#
#  Copyright 2014 Applied Invention, LLC.
#-------------------------------------------------------------------

'''The module containing the ClassJsonTypescriptWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .ClassJsonClassDesc import ClassJsonClassDesc
from .ClassJsonFieldDesc import ClassJsonFieldDesc
from .jsonTypes.JsonObj import JsonObj
from .jsonTypes.JsonObjRegistry import JsonObjRegistry
from typing import List
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ClassJsonTypescriptWriter:
  '''Writes the typescript for a ClassJson class.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new ClassJsonTypescriptWriter.
    '''

    pass

  #-----------------------------------------------------------------
  def writeAll(self,
               jsonObjs: List[JsonObj],
               interfaceImports: bool,
               indent: int = 0) -> Tuple[str, str]:
    '''Writes the typescript for the specified list of class.

    The generated classes are wrapped in a namespace.

    @param jsonObjs The classes whose Typescript will be written.
    @param interfaceImports Whether to add interface includes to the class file.
    @param indent How many spaces to indent the returned text.

    @return a string containing the Typescript description.
    '''

    interfaceHeader = '''\
//------------------------------------------------------------------
//  classJsonInterfaces.ts
//
// This file contains generated Typescript interfaces for all ClassJson
// classes used in web actions.
//
//------------------------------------------------------------------

// Allow this file to break the 80-char-max-line-length rule.
/* tslint:disable:max-line-length */

// Allow this file to declare members in constructor argument lists.
/* tslint:disable:no-parameter-properties */

import { Day } from '../axe/date/Day';
import { Duration } from '../axe/date/Duration';
import { Map } from '../axe/util/Map';

'''

    classHeader = '''\
//------------------------------------------------------------------
//  classJsonClasses.ts
//
// This file contains private implementations of ClassJson definitions
// used in web actions.
// Do not use these.  Instead use the public interfaces
// in classJsonInterfaces.ts
//
//------------------------------------------------------------------

// Allow this file to break the 80-char-max-line-length rule.
/* tslint:disable:max-line-length */

// Allow this file to declare members in constructor argument lists.
/* tslint:disable:no-parameter-properties */

import { Day } from '../axe/date/Day';
import { Duration } from '../axe/date/Duration';
import { ClassJsonRegistry } from '../axe/classJson/ClassJsonRegistry';
import { ClassJsonDesc } from '../axe/classJson/ClassJsonDesc';
import { JsonAny } from '../axe/classJson/jsonTypes/JsonAny';
import { JsonDate } from '../axe/classJson/jsonTypes/JsonDate';
import { JsonDateTime } from '../axe/classJson/jsonTypes/JsonDateTime';
import { JsonDuration } from '../axe/classJson/jsonTypes/JsonDuration';
import { JsonLink } from '../axe/classJson/jsonTypes/JsonLink';
import { JsonList } from '../axe/classJson/jsonTypes/JsonList';
import { JsonMap } from '../axe/classJson/jsonTypes/JsonMap';
import { JsonObj } from '../axe/classJson/jsonTypes/JsonObj';
import { JsonPrimitiveType } from '../axe/classJson/jsonTypes/JsonPrimitiveType';
import { Map } from '../axe/util/Map';

'''

    interfaceTexts = []
    classTexts = []
    importedClasses: List[str] = []

    for jsonObj in jsonObjs:

      desc = jsonObj.desc
      className = desc.className
      properties = desc.properties

      superclasses = JsonObjRegistry.superclasses(jsonObj)
      superclassNames = [x.desc.className for x in superclasses]

      iText = self.writeInterface(className, superclassNames, properties, '',
                                  indent)
      classText = self.writeClass(className, '', 'Basic%s',
                                  properties, '', indent)

      subclasses: List[JsonObj] = JsonObjRegistry.subclasses(jsonObj)
      allClasses = [jsonObj] + subclasses
      importedClasses.extend([x.desc.className for x in allClasses])

      interfaceTexts.append(iText)
      classTexts.append(classText)

    # Build up the imports for ClassJson classes used in the actions.

    importedText = ''

    if interfaceImports:

      # Remove duplicates.
      importedClasses = list(set(importedClasses))

      importedClasses.sort()

      importStr = "import { %s } from './classJsonInterface';"

      importLines = [importStr % clazz for clazz in importedClasses]
      importedText = '\n'.join(importLines) + '\n\n'

    # Build up the text to register the classes.

    classNames = [jsonObj.desc.className for jsonObj in jsonObjs]
    registerText = self.writeRegister(classNames, 'Basic%s', indent)
    registerDescText = self.writeRegisterDescs(jsonObjs, indent)

    interfaceText = interfaceHeader + '\n'.join(interfaceTexts) + "\n"

    classText = classHeader + importedText + '\n'.join(classTexts) + "\n"
    classText += '\n\n' + registerText
    classText += '\n\n' + registerDescText

    return interfaceText, classText

  #-----------------------------------------------------------------
  def writeRegister(self,
                    classNames: List[str],
                    concreteTemplate: str,
                    indent: int) -> str:
    '''Writes the typescript to register the specified classes.

    @param className A string list of names for which register calls will be
                     written.
    @param concreteTemplate A string template for converting from className
                            to a concrete class name.  Must contain a single
                            '%s' character where the className will be added.
    @param indent How many spaces to indent the returned text.

    @return a string containing the Typescript register function.
    '''

    # The correct number of spaces to start each line with.
    indentStr = ' ' * indent * 2

    # The correct number of spaces to start each function line with.
    iStr = ' ' * (indent + 1) * 2

    text = ''

    for className in classNames:
      concreteClassName = concreteTemplate % (className,)

      text += iStr
      text += "ClassJsonRegistry.registry.register('" + className
      text += "',\n"
      text += iStr
      text += "                                    " + concreteClassName
      text += ",\n"
      text += iStr
      text += "                                    false);\n"

    func = indentStr + "export function registerDefaults() : void\n"
    func += indentStr + "{\n"
    func += text
    func += indentStr + "}\n"

    return func

  #-----------------------------------------------------------------
  def writeRegisterDescs(self, jsonObjs: List[JsonObj], indent: int) -> str:
    '''Writes the typescript to register the specified classes.

    @param jsonObjs The classes whose description will be written.
    @param indent How many spaces to indent the returned text.

    @return a string containing the Typescript register function.
    '''

    # The correct number of spaces to start each line with.
    indentStr = ' ' * indent * 2

    # The correct number of spaces to start each function line with.
    iStr = ' ' * (indent + 1) * 2

    text = ''

    for jsonObj in jsonObjs:
      text += self.writeRegisterDesc(jsonObj.desc, indent + 1)

    func = indentStr + "export function registerDescs() : void\n"
    func += indentStr + "{\n"
    func += iStr + "let desc: ClassJsonDesc = null;\n"
    func += text
    func += indentStr + "}\n"

    return func

  #-----------------------------------------------------------------
  def writeRegisterDesc(self,
                        classJsonDesc: ClassJsonClassDesc,
                        indent: int) -> str:
    '''Writes the typescript to register the specified classes.

    @param classJsonDesc The class whose description will be written.
    @param indent How many spaces to indent the returned text.

    @return a string containing the Typescript register function.
    '''

    # The correct number of spaces to start each line with.
    indentStr = ' ' * indent * 2

    text = ''

    text += '\n'
    text += indentStr
    text += 'desc = new ClassJsonDesc("%s");\n' % classJsonDesc.className

    for classJsonField in classJsonDesc.properties:
      fieldName = classJsonField.name
      jsonType = classJsonField.jsonType
      label = jsonType.toTypescriptDesc()

      text += indentStr
      text += 'desc.addField("%s", %s);\n' % (fieldName, label)

    text += indentStr
    text += 'ClassJsonRegistry.registry.addDesc(desc);\n'

    return text

  #-----------------------------------------------------------------
  def write(self,
            clazz: type,
            indent: int = 0) -> str:
    '''Writes the typescript for the specified class.

    @param clazz The class whose Typescript will be written.
                 Must be decorated with @ClassJsonClass.
    @param indent How many spaces to indent the returned text.

    @return a string containing the Typescript description.
    '''

    jsonObj = JsonObjRegistry.getForClass(clazz)
    desc = jsonObj.desc

    superclasses: List[JsonObj] = JsonObjRegistry.superclasses(jsonObj)
    superclassNames = [x.desc.className for x in superclasses]

    ret = self.writeInterface(desc.className, superclassNames, desc.properties,
                              '', indent)
    ret += "\n"
    ret += self.writeClass(desc.className, '', 'Basic%s',
                           desc.properties, '', indent)

    return ret

  #-----------------------------------------------------------------
  def writeInterface(self,
                     className: str,
                     superclassNames: List[str],
                     properties: List[ClassJsonFieldDesc],
                     propertyNamespace: str,
                     indent: int) -> str:
    '''Writes the typescript interface for the specified class.

    @param className The name of the class being written.
    @param properties List of ClassJsonField objects.  The properties
                      whose Typescript will be written.
    @param propertyNamespace The namespace that member variable classes
                              will be found in.  May be '' for no namespace.
    @param indent How many spaces to indent the returned text.

    @return a string containing the interface description.
    '''

    assert className is not None

    # The correct number of spaces to start each line with.
    indentStr = ' ' * indent * 2

    propLines = []

    for propDesc in properties:
      line = ('  ' +
              propDesc.name + ': ' +
              propDesc.jsonType.toTypescriptLabel(propertyNamespace) + ';')
      propLines.append(line)

    superclassStr = ""
    if superclassNames:
      superclassStr = " extends " + ', '.join(superclassNames)

    lines = [
      'export interface ' + className + superclassStr,
      '{',
      '}'
      ]

    # Insert the properties between the braces.
    lines = lines[:-1] + propLines + lines[-1:]

    text = indentStr + ('\n' + indentStr).join(lines) + '\n'

    return text

  #-----------------------------------------------------------------
  def writeClass(self,
                 className: str,
                 interfaceNamespace: str,
                 concreteTemplate: str,
                 properties: List[ClassJsonFieldDesc],
                 propertyNamespace: str,
                 indent: int) -> str:
    '''Writes the typescript class for the specified class.

    @param className The name of the class being written.
    @param interfaceNamespace The namespace that the implemented interface
                              will be found in.  May be '' for no namespace.
    @param concreteTemplate A string template for converting from className
                            to a concrete class name.  Must contain a single
                            '%s' character where the className will be added.
    @param properties List of ClassJsonField objects.  The properties
                      whose Typescript will be written.
    @param propertyNamespace The namespace that member variable classes
                              will be found in.  May be '' for no namespace.
    @param indent How many spaces to indent the returned text.

    @return a string containing the class description.
    '''

    assert className is not None

    interfaceName = className
    if interfaceNamespace:
      interfaceName = interfaceNamespace + '.' + className

    className = concreteTemplate % (className,)

    # Generate declaration and constructor.
    lines = []

    for propDesc in properties:
      line = ('public ' + propDesc.name + ': ' +
              propDesc.jsonType.toTypescriptLabel(propertyNamespace))
      lines.append(line)

    indentStr = ' ' * ((indent * 2) + len('  constructor('))
    propsStr = (',\n' + indentStr).join(lines)

    indentStr = ' ' * indent * 2
    ret = ''
    ret += indentStr + 'class ' + className + '\n'
    ret += indentStr + '{\n'
    ret += indentStr + '  constructor(' + propsStr + ')\n'
    ret += indentStr + '  {\n'
    ret += indentStr + '  }\n'

    # Generate fromJson() method.
    lines = []

    for propDesc in properties:
      line = 'src.' + propDesc.name
      lines.append(line)

    indentStr = ' ' * ((indent * 2) + len('    return new ' + className + '('))
    propsStr = (',\n' + indentStr).join(lines)

    indentStr = ' ' * indent * 2
    ret += '\n'
    ret += indentStr + '  '
    ret += 'static fromJson(src: ' + interfaceName + ') : '  + className + '\n'
    ret += indentStr + '  {\n'
    ret += indentStr + '    return new ' + className + '(' + propsStr + ');\n'
    ret += indentStr + '  }\n'
    ret += '\n'

    # Generate toJson() method and then close out the class
    ret += indentStr
    ret += '  static toJson(src: ' + className + ') : ' + interfaceName + '\n'
    ret += indentStr + '  {\n'
    ret += indentStr + '    return src;\n'
    ret += indentStr + '  }\n'
    ret += indentStr + '}\n'

    return ret
