#-------------------------------------------------------------------
#  ClassJsonSchemaWriter.py
#
#  The ClassJsonSchemaWriter class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ClassJsonSchemaWriter class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from collections import OrderedDict
from .ClassJsonFieldDesc import ClassJsonFieldDesc
from .jsonTypes.JsonObjRegistry import JsonObjRegistry
from .jsonTypes import JsonList
from .jsonTypes import JsonObj
from typing import Dict
from typing import List
from typing import Optional
from typing import Tuple
import json
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ClassJsonSchemaWriter:
  '''Writes the schema for a ClassJson class.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new ClassJsonSchemaWriter.
    '''

    pass

  #-----------------------------------------------------------------
  def writeClass(self,
                 clazz: type,
                 indent: int = 0,
                 formatForHtml: bool = False) -> str:
    '''Writes the schema for the specified class.

    Any nested classes are included following the first class.

    @param clazz The class whose ClassJson schema will be written.
                 Must be decorated with @ClassJsonClass.
    @param indent How many spaces to indent the returned text.

    @return a string containing the schema description.
    '''

    jsonObj = JsonObjRegistry.getForClass(clazz)
    desc = jsonObj.desc
    return self.writeAllClasses(jsonObj,
                                desc.className,
                                desc.properties,
                                indent,
                                formatForHtml)

  #-----------------------------------------------------------------
  def writeFields(self,
                  className: Optional[str],
                  properties: List[ClassJsonFieldDesc],
                  indent: int = 0,
                  formatForHtml: bool = False) -> str:
    '''Writes the schema for the specified property list.

    Any nested classes are included following the first class.

    @param className The name of the class being written.
                     If None, the _class pseudo-property will be omitted.
    @param properties List of ClassJsonField objects.  The properties
                      whose ClassJson schema will be written.
    @param indent How many spaces to indent the returned text.

    @return a string containing the schema description.
    '''

    return self.writeAllClasses(None, className, properties, indent,
                                formatForHtml)

  #-----------------------------------------------------------------
  def writeAllClasses(self,
                      jsonObj: Optional[JsonObj],
                      className: Optional[str],
                      properties: List[ClassJsonFieldDesc],
                      indent: int = 0,
                      formatForHtml: bool = False) -> str:
    '''Writes the schema for the specified class and properties.

    Any nested classes are included following the first class.

    @param clazz The class whose ClassJson schema will be written.
                 Must be decorated with @ClassJsonClass.
                 May be None.  If not None, must be the class whose
                 name and properties are specfied.
    @param className The name of the class being written.
                     If None, the _class pseudo-property will be omitted.
    @param properties List of ClassJsonField objects.  The properties
                      whose ClassJson schema will be written.
    @param indent How many spaces to indent the returned text.

    @return a string containing the schema description.
    '''

    if jsonObj:

      assert jsonObj.desc.className == className
      assert jsonObj.desc.properties == properties

    # The string we'll return.
    ret = ''

    # The classes remaining to write.
    classesToWrite: List[Tuple[Optional[JsonObj], int]] = [(jsonObj, indent)]

    # Keep track of written classes so we don't repeat.
    writtenClasses: List[JsonObj] = []

    while classesToWrite:

      jsonObj, indent = classesToWrite.pop(0)

      # Don't repeat any classes.
      if jsonObj in writtenClasses:
        continue

      # Write the next class in line.
      if ret:
        ret += '\n'
      if jsonObj:
        ret += self.writeJsonObj(jsonObj, indent, formatForHtml)
        writtenClasses.append(jsonObj)
      else:
        ret += self.writeClassFields(className,
                                     properties,
                                     indent,
                                     formatForHtml)

      # Add any child classes to the list to write.
      if jsonObj:
        properties = jsonObj.desc.properties
      for propDesc in properties:

        jsonType = propDesc.jsonType

        # For lists, look at the item type.
        while isinstance(jsonType, JsonList):
          jsonType = jsonType.itemType

        if isinstance(jsonType, JsonObj):
          classesToWrite.append((jsonType, indent + 1))

    return ret

  #-----------------------------------------------------------------
  def writeJsonObj(self,
                   jsonObj: JsonObj,
                   indent: int = 0,
                   formatForHtml: bool = False) -> str:
    '''Writes the schema for the specified class.

    @param clazz The class whose ClassJson schema will be written.
                 Must be decorated with @ClassJsonClass.
    @param indent How many spaces to indent the returned text.

    @return a string containing the schema description.
    '''

    desc = jsonObj.desc

    return self.writeClassFields(desc.className,
                                 desc.properties,
                                 indent,
                                 formatForHtml)

  #-----------------------------------------------------------------
  def writeClassFields(self,
                       className: Optional[str],
                       properties: List[ClassJsonFieldDesc],
                       indent: int = 0,
                       formatForHtml: bool = False) -> str:
    '''Writes the schema for the specified class.

    @param className The name of the class being written.
                     If None, the _class pseudo-property will be omitted.
    @param properties List of ClassJsonField objects.  The properties
                      whose ClassJson schema will be written.
    @param indent How many spaces to indent the returned text.
    @param formatForHtml if the class will be inserted directly into html,
                          if true we must escape < and > character

    @return a string containing the schema description.
    '''

    lines = []
    if formatForHtml:
      leftBrace = '&#123;'
      rightBrace = '&#125;'
      indentStr = '<td></td>'
      lessThan = '&lt;'
      greaterThan = '&gt;'

      for propDesc in properties:
        line = ('</td><td>\"' + propDesc.name + '\" : ' + lessThan +
                propDesc.jsonType.toLabel() + greaterThan)
        lines.append(line)

      lines[0] = '<td>' + lines[0]
      propsStr = ',</td></tr><tr><td>'.join(lines)
      propsStr += '</td></tr>'

      ret = '<tr>' + indentStr + '<td>' + leftBrace + '</td></tr><tr>'
      if className is not None:
        ret += indentStr + '<td>\"_class\" : ' + className  + ',</td><tr>'
      ret += propsStr + '<tr>' + indentStr + '<td>' + rightBrace + '</td></tr>'

      # Format for readability.
      ret = ret.replace('<tr>', '<tr>\n')
      ret = ret.replace('</tr>', '</tr>\n')
      ret = ret.replace('<td>', '  <td>')
      ret = ret.replace('</td>', '</td>\n')

    else:
      for propDesc in properties:
        line = ('"' + propDesc.name + '" : <' +
                propDesc.jsonType.toLabel() + '>')
        lines.append(line)

      propsStr = ',\n  '.join(lines)

      ret = '{\n  '
      if className is not None:
        ret += '"_class" : "' + className  + '",\n  '
      ret += propsStr + '\n}'

      # Apply the indent.
      indentStr = ' ' * indent * 2
      ret = indentStr + ret
      ret = ret.replace('\n', '\n' + indentStr)

    return ret

  #-----------------------------------------------------------------
  def writeAllJson(self) -> str:
    '''Returns a JSON string describing all known classes.
    '''

    return self.writeJsonObjDictToJson(JsonObjRegistry.registeredNames)

  #-----------------------------------------------------------------
  def writeJsonObjDictToJson(self, classDict: Dict[str, JsonObj]) -> str:
    '''Returns a JSON string describing all classes in a dictionary.
    '''

    allClasses: Dict[str, JsonObj] = classDict
    classNames: List[str] = sorted(allClasses.keys())

    classesDict: Dict[str, Dict[str, str]] = OrderedDict()

    for className in classNames:
      classesDict[className] = self.writeJsonObjToJson(allClasses[className])

    return json.dumps(classesDict, indent=2)

  #-----------------------------------------------------------------
  def writeJsonObjToJson(self, jsonObj: JsonObj) -> Dict[str, str]:
    '''Returns a JSON-ready dictionary describing the specified class.
    '''

    desc = jsonObj.desc

    classDict: Dict[str, str] = OrderedDict()

    for prop in desc.properties:
      classDict[prop.name] = prop.jsonType.toLabel()

    return classDict
