#-------------------------------------------------------------------
#  ClassJsonLink.py
#
#  The ClassJsonLink class.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ClassJsonLink class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ClassJsonLink:
  '''Attribute that is a pointer to an object somewhere else in the JSON tree.

  When a single object appears in multiple places in a JSON tree,
  one of those places will be declared as the class type itself.
  All the other places the object appears must be a ClassJsonLink.

  For example, suppose a company has a list of employees.  One of those
  employees is employee of the month:

    class Company:

       employees: List[Employee]
       employeeOfTheMonth: Employee

  What should the @ClassJsonClass look like?  If we do the obvious thing:

    @ClassJsonClass([ClassJsonField('employees', [Employee]),
                     ClassJsonField('employeeOfTheMonth', Employee)]

  then a decoded Company object will end up with two copies of the
  Employee object who is employeeOfTheMonth:  one in the employees list,
  and one in the employeeOfTheMonth variable.  But that's not what we want.
  We want employeeOfTheMonth to point to one of the objects in the
  employees list.

  ClassJsonLink makes that happen:

    @ClassJsonClass([ClassJsonField('employees', [Employee]),
                     ClassJsonField('employeeOfTheMonth',
                                    ClassJsonLink(Employee, 'employees', 'id')]

  The ClassJsonLink says: the variable 'employeeOfTheMonth' is a pointer
  to an object of type Employee which is found in the 'employees'
  collection whose 'id' member variable can be used to identify it.
  '''

  #-----------------------------------------------------------------
  def __init__(self, clazz: type, path: str, idFields: List[str]) -> None:
    '''Creates a new ClassJsonLink.
    '''

    # The class of the object this is a link to.  Must be a ClassJsonClass.
    self.objClass = clazz

    # The path to find the object this is a link to.
    self.objPath: str = path

    # The member variable to use to find the correct object in the collection.
    self.idFields: List[str] = idFields

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['objClass', 'objPath', 'idFields']

    return StringUtil.formatRepr(self, attrs)
