#-------------------------------------------------------------------
#  ClassJsonDecoder.py
#
#  The ClassJsonDecoder class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ClassJsonDecoder class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .jsonTypes.JsonObjRegistry import JsonObjRegistry
from .ClassJsonException import ClassJsonException
import json
from typing import Optional
from typing import Type
from typing import TypeVar
from .jsonTypes import JsonTypeError
#
# Import statements go above this line.
#-------------------------------------------------------------------

#===================================================================
# Generic class that JSON is decoded into an object of.
TargetType = TypeVar('TargetType')

#===================================================================
class ClassJsonDecoder:
  '''Decodes JSON into an object.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new ClassJsonDecoder.
    '''

    pass

  #-----------------------------------------------------------------
  def decode(self, jsonStr: str, targetClass: Type[TargetType]) -> TargetType:
    '''Decodes the specified JSON string into an object.

    @param jsonStr The string to be decoded.
    @param targetClass The class that is being decoded.

    @return A @ClassJsonClass-decorated object of the specified class.
    '''

    obj = self.decodeObject(jsonStr)

    if not isinstance(obj, targetClass):
      msg = ("Error decoding JSON.  Expected an object of type %s, " +
             "but the JSON has an object of type %s.")
      msg = msg % (targetClass, type(obj))
      raise ClassJsonException(msg)

    return obj

  #-----------------------------------------------------------------
  def decodeObject(self, jsonStr: str) -> object:
    '''Decodes the specified JSON string into an object.

    @param jsonStr The string to be decoded.

    @return A @ClassJsonClass-decorated object.
    '''

    objDict = json.loads(jsonStr)

    if '_class' not in objDict:
      msg = 'Root object is missing "_class" attribute.  Json: %s'
      msg = msg % jsonStr
      raise ClassJsonException(msg)

    classJsonName: str = objDict['_class']

    if not JsonObjRegistry.nameIsRegistered(classJsonName):
      msg = 'Root object _class "%s" has not been registered.  '
      msg += 'Unable to decode JSON: %s'
      msg = msg % (classJsonName, jsonStr)
      raise ClassJsonException(msg)

    jsonType = JsonObjRegistry.getForName(classJsonName)

    errorMsg: Optional[JsonTypeError] = jsonType.validateJson(objDict)
    if errorMsg:
      msg = errorMsg.format()
      msg += "\n\nUnable to decode JSON:\n%s" % jsonStr

      raise ClassJsonException(msg)

    obj = jsonType.decode(objDict)

    obj = jsonType.decodeLinks([], obj)

    return obj
