#-------------------------------------------------------------------
#  ClassJsonClassDesc.py
#
#  The ClassJsonClassDesc class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ClassJsonClassDesc class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .ClassJsonFieldDesc import ClassJsonFieldDesc
from typing import Any
from typing import Callable
from typing import List
from typing import Optional
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ClassJsonClassDesc:
  '''Description of a class to be used with classJson.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               className: str,
               properties: List[ClassJsonFieldDesc],
               ctor: Optional[Callable[..., Any]],
               ctorArgs: bool,
               encodeOnly: bool,
               isAbstract: bool) -> None:
    '''Creates a new ClassJsonClassDesc.
    '''

    # The name of the class as it will appear in _class JSON attributes.
    self.className: str = className

    # The serializable fields for the decorated class.
    self.properties: List[ClassJsonFieldDesc] = properties

    # The constructor function to use to create objects during decoding.
    self.ctor: Optional[Callable[..., Any]] = ctor

    # Whether to pass args into the constructor or set properties.
    self.ctorArgs: bool = ctorArgs

    # If True, an object can be converted to JSON, but not JSON -> object.
    self.encodeOnly: bool = encodeOnly

    # If True, this class is a base class. Only derived classes can be created.
    self.isAbstract: bool = isAbstract

  #----------------------------------------------------------------
  def hasProperty(self, name: str) -> bool:
    '''Returns true if there is a properity with the specified name.
    '''

    for prop in self.properties:
      if prop.name == name:
        return True

    return False

  #----------------------------------------------------------------
  def getProperty(self, name: str) -> ClassJsonFieldDesc:
    '''Returns the properity with the specified name.
    '''

    for prop in self.properties:
      if prop.name == name:
        return prop

    raise ValueError("Unknown name: " + str(name))

  #----------------------------------------------------------------
  def __repr__(self):
    '''Returns a string representation of this object
    '''

    attrs = [
      'className',
      'properties',
      'ctor',
      'ctorArgs',
      'encodeOnly',
      'isAbstract',
    ]

    return StringUtil.formatRepr(self, attrs)
