#-------------------------------------------------------------------
#  ClassJsonClass.py
#
#  The ClassJsonClass class.
#
#  Copyright 2016 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ClassJsonClass class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .ClassJsonClassDesc import ClassJsonClassDesc
from .ClassJsonField import ClassJsonField
from .ClassJsonFieldDesc import ClassJsonFieldDesc
from .jsonTypes.JsonObjRegistry import JsonObjRegistry
from .jsonTypes import JsonObj
from .ClassJsonException import ClassJsonException
from typing import Any
from typing import Callable
from typing import List
from typing import Optional
import inspect
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ClassJsonClass:
  '''Decorator that marks a class to be used with classJson.

  For example:

  @ClassJsonClass([ClassJsonField('name2', 'string'),
                   ClassJsonField('x2', 'int'),
                   ClassJsonField('y2', 'int')])
  class TestClass3:
    pass


  A list of ClassJsonField object must be provided.

  If the decoder should a factory function rather than the __init__() function,
  you can specify the factory function with the 'ctor' argument.

  If the object will be encoded into JSON string but never decoded,
  you can specifiy encodeOnly=True.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               properties: List[ClassJsonField],
               ctor: str = None,
               encodeOnly: bool = False,
               isAbstract: bool = False) -> None:
    '''Creates a new ClassJsonClass.
    '''

    # The serializable fields for the decorated class.
    self.properties: List[ClassJsonField] = properties

    # The function that should be used to create new objects.
    # If 'None' the normal constructor (__init__) will be used.
    self.ctorName: Optional[str] = ctor

    # If True, an object can be converted to JSON, but not JSON -> object.
    self.encodeOnly: bool = encodeOnly

    # If True, this class is a base class. Only derived classes can be created.
    self.isAbstract: bool = isAbstract

  #-----------------------------------------------------------------
  def decorate(self, clazz: type) -> type:
    '''Sets up a class to be read/written by ClassJson.

    @param clazz The class to set up.
    '''

    # The constructor function to use to create objects during decoding.
    ctor: Optional[Callable[..., Any]] = None

    # Set the constructor if a name was provided.
    if self.ctorName:
      if not hasattr(clazz, self.ctorName):
        msg = ('Error setting up class ' + clazz.__name__ + '.  The provided ' +
               'ctor name ' + str(self.ctorName) + ' must be a static method.')
        raise ClassJsonException(msg)

      ctor = getattr(clazz, self.ctorName)

    # If the constructor is a static method, get the callable function.
    if isinstance(ctor, staticmethod):
      ctor = ctor.__func__

    properties: List[ClassJsonFieldDesc] = [x.toDesc() for x in self.properties]

    # Whether to pass args into the constructor or set properties.
    ctorArgs: bool = False

    if not self.encodeOnly and not self.isAbstract:
      ctorArgs = self.checkCtor(clazz, properties, ctor)

    # The name of the encoded class.
    # Use the name of the class with no package.
    className: str = clazz.__name__.split('.')[-1]

    # Create a description object.
    desc = ClassJsonClassDesc(className,
                              properties,
                              ctor,
                              ctorArgs,
                              self.encodeOnly,
                              self.isAbstract)

    # Put the class in the global registry.

    jsonObj = JsonObj(clazz, desc)
    JsonObjRegistry.register(jsonObj)

    return clazz

  #-----------------------------------------------------------------
  def __call__(self, clazz: type) -> type:
    '''Sets up a class to be read/written by ClassJson.

    @param clazz The class to set up.
    '''

    return self.decorate(clazz)

  #-----------------------------------------------------------------
  def checkCtor(self,
                clazz: type,
                properties: List[ClassJsonFieldDesc],
                ctor: Optional[Callable]):
    '''Checks that the class has a contructor that can be called with the props.

    @param clazz The class whose constructor will be checked.
    @param properties The list of properties to check against.
    @param ctor The constructor function specified by the user.

    @return False if the no-arg ctor should be used, True if with arguments.
    '''

    if isinstance(ctor, staticmethod):
      ctor = ctor.__func__

    if ctor:
      fullArgSpec = inspect.getfullargspec(ctor)
      args = fullArgSpec.args

    else:
      classInit = clazz.__init__ # type: ignore
      fullArgSpec = inspect.getfullargspec(classInit)
      args = fullArgSpec.args

      # Ignore the 'self' arg.
      args = args[1:]

    expectedNames = []
    for prop in properties:
      expectedNames.append(prop.name)

    if args and expectedNames != args:
      ctorName = '__init__'
      if ctor:
        ctorName = ctor.__name__
      msg = ("Error setting up " + clazz.__name__  + " for classJson.  " +
             "The decorator lists arguments that are different from " +
             "constructor function " + ctorName + '.  Decorator args: ' +
             str(expectedNames) + '  ' + ctorName + ' args: ' + str(args))
      raise ClassJsonException(msg)

    return bool(args)

  #----------------------------------------------------------------
  def __repr__(self):
    '''Returns a string representation of this object
    '''

    attrs = [
      'properties',
      'ctorName',
      'encodeOnly',
      'isAbstract',
    ]

    return StringUtil.formatRepr(self, attrs)
