#-------------------------------------------------------------------
#  Targets.py
#
#  The Targets module.
#
#  Copyright 2013  Applied Invention, LLC.
#-------------------------------------------------------------------

"""The Targets class.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
from typing import List
from typing import Tuple
#
# Import statements go above this line.
#-------------------------------------------------------------------

# A type that is a tuple of three strings.
Str3 = Tuple[str, str, str]

#===================================================================
class Targets:
  '''Class for managing the list of targets that the user has requested.
  '''

  #-------------------------------------------------------------------
  def __init__(self, targets: List[str]) -> None:
    """Create a new Targets.

    @param targets A list of strings the user passed in on the command line.
    """

    self.targets: List[str] = targets

    # If '.' or no targets are given on the command line, then we're building
    # everything.
    allTargets = not targets or '.' in targets

    self.allTests = ('test' in targets or
                     'build/unittest' in targets or
                     allTargets)
    self.tests: List[str] = []
    for target in targets:
      if target.startswith('test-'):
        testName = target[len('test-'):]
        self.tests.append(testName)

    self.allCoverTests: bool = ('cover' in targets or
                                'cover-report' in targets or
                                'build/coverage' in targets or
                                allTargets)
    self.coverTests: List[str] = []
    for target in targets:
      if target.startswith('cover-test-'):
        testName = target[len('cover-test-'):]
        self.coverTests.append(testName)
      if target.startswith('cover-report-'):
        testName = target[len('cover-report-'):]
        self.coverTests.append(testName)

    # If an explicit path to something in the test directories is given,
    # then create all the targets in that directory.
    for target in targets:
      if target.startswith('build/unittest'):
        self.allTests = True
      elif target.startswith('build/coverage'):
        self.allCoverTests = True

  #-------------------------------------------------------------------
  def testsToBuild(self, testList: List[Str3]) -> List[Str3]:
    '''Given a list of available tests, returns those the user wants to run.
    '''

    if self.allTests:
      return testList

    testsToBuild: List[Tuple[str, str, str]] = []
    for testClassTuple in testList:
      classFile, testClassName, testedClassName = testClassTuple
      bareTestClassName = testClassName.split('.')[-1]
      bareTestedClassName = testedClassName.split('.')[-1]
      packageName = '.'.join(testedClassName.split('.')[:-1])

      # Supress unused variable warning.
      classFile = classFile # pylint: disable=self-assigning-variable

      for target in self.tests:
        if target in (testClassName,
                      testedClassName,
                      bareTestClassName,
                      bareTestedClassName,
                      packageName):
          testsToBuild.append(testClassTuple)

    return testsToBuild

  #-------------------------------------------------------------------
  def coverTestsToBuild(self, testList: List[Str3]) -> List[Str3]:
    '''Given a list of available tests, returns those the user wants to run.
    '''

    if self.allCoverTests:
      return testList

    testsToBuild = []
    for testClassTuple in testList:
      classFile, testClassName, testedClassName = testClassTuple
      bareTestClassName = testClassName.split('.')[-1]
      bareTestedClassName = testedClassName.split('.')[-1]
      packageName = '.'.join(testedClassName.split('.')[:-1])

      # Supress unused variable warning.
      # pylint: disable=self-assigning-variable
      classFile = classFile

      for target in self.coverTests:
        if target in (testClassName,
                      testedClassName,
                      bareTestClassName,
                      bareTestedClassName,
                      packageName):
          testsToBuild.append(testClassTuple)

    return testsToBuild
