#-------------------------------------------------------------------
#  TarGz.py
#
#  The TarGz module.
#
#  Copyright 2013  Applied Invention, LLC.
#-------------------------------------------------------------------

"""A .tar.gz builder for SCons.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
import os.path
import SCons
from SCons.Environment import Base as SconsEnvironment
from SCons.Node import NodeList
from typing import List
#
# Import statements go above this line.
#-------------------------------------------------------------------

def tarGzCmd(env: SconsEnvironment,
             tarFileBase: str,
             tarDir: str,
             tarFiles: List[str]) -> NodeList:
  '''Returns a command that will create a .tar version of a directory.

  @param tarFileBase The base name (without the .tar extension) of the
                     tar file that will be generated.
  @param tarDir The directory to be tarred up.
  @param tarFiles The files to be tarred up.
  '''

  tar = tarCmd(env, tarFileBase, tarDir, tarFiles)
  tarGz = env.Gzip(tar)

  return tarGz

#-------------------------------------------------------------------
def tarCmd(env: SconsEnvironment,
           tarFileBase: str,
           tarDir: str,
           tarFiles: List[str]) -> NodeList:
  '''Returns a command that will create a .tar version of a directory.

  @param tarFileBase The base name (without the .tar extension) of the
                     tar file that will be generated.
  @param tarDir The directory to be tarred up.
  @param tarFiles The files to be tarred up.
  '''

  # I tried to implement this as a real builder, but couldn't figure out
  # how to do Clean(tarFile, tarDir) with a builder.

  # Tar the files.
  tarFile = tarFileBase + ".tar"
  parentDir, dirOnly = os.path.split(tarDir)
  tarCmdStr = "tar cf " + tarFile + " -C " + parentDir + " " + dirOnly
  action = env.Action(tarCmdStr, "Creating tar file " + tarFile + "...")
  tar = env.Command(tarFile, tarFiles, action)

  env.Clean(tar, tarDir)

  return tar

#-------------------------------------------------------------------
def addGzipBuilder(env: SconsEnvironment) -> None:
  '''Adds a builder to the specified environment.'''

  # Using -c and redirecting stdout seems to be the only way to get
  # gzip not to delete the original file that it is zipping.
  cmd = "gzip -c $SOURCE > $TARGET"

  action = SCons.Action.Action(cmd, "Zipping ${TARGET.file}...")
  builder = SCons.Builder.Builder(action=action,
                                  suffix='.tar.gz')

  env.Append(BUILDERS={'Gzip': builder})
