#-------------------------------------------------------------------
#  PyTest.py
#
#  The PyTest module.
#
#  Copyright 2006  Applied Invention, LLC.
#-------------------------------------------------------------------

"""A unittest Builder for SCons.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
from .Util import silent
from multiprocessing import Queue
from SCons.Environment import Base as SconsEnvironment
from SCons.Node import Node
from typing import List
from unittest import TestSuite
import SCons
import ai.axe.build.sconstruct.Coverage
import coverage
import importlib
import multiprocessing
import os.path
import sys
import unittest
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def addPyTestBuilder(env: SconsEnvironment):
  """Adds the PyTest builder to the specified environment.

  Call this function with your SCons.Environment if you want to build
  unittests.  After calling this function, you can build unit test
  runs like so:

     env.PyTest('build/unittest/mytest.passed', 'src/mytest.py')
     env.PyCov('build/cover/mytest.cov-passed', 'src/mytest.py')

  @param env An SCons.Environment object.
  """

  # The PyTest builder.

  actions = [
    SCons.Action.Action(runUnitTest, "Running test ${TARGET.file}..."),
    SCons.Action.Action('touch $TARGET', silent),
  ]

  builder = SCons.Builder.Builder(action=actions)
  env.Append(BUILDERS={'PyTest': builder})

  # The PyCov builer.

  actions = [
    SCons.Action.Action(runCovTestSub, "Running test ${TARGET.file}..."),
    SCons.Action.Action('touch $TARGET', silent),
  ]

  builder = SCons.Builder.Builder(action=actions)
  env.Append(BUILDERS={'PyCov': builder})

#-------------------------------------------------------------------
def runUnitTest(target: List[Node], source: List[Node], env: SconsEnvironment):
  """Run a unit test.
  """

  assert len(source) == 1
  assert len(target) == 1

  srcFile: str = str(source[0])

  targetFile: str = str(target[0])
  assert targetFile.endswith('.passed')

  # SCons for some reason imports cPickle and replaces the built-in
  # pickle with it, which Pandas chokes on when trying to inherit from pickle.

  cPickleModule = sys.modules['pickle']
  moduleName = 'pickle'
  pickleModule = importlib.import_module(moduleName)
  sys.modules['pickle'] = pickleModule

  suite = buildTestSuite(srcFile, env)
  runner = unittest.TextTestRunner(verbosity=2)
  result = runner.run(suite)

  # Replace pickle with cPickle as SCons likes.
  sys.modules['pickle'] = cPickleModule

  # Return non-zero to make the build fail.
  return result.errors + result.failures

#-------------------------------------------------------------------
def runCovTestSub(target: List[Node],
                  source: List[Node],
                  env: SconsEnvironment) -> int:
  '''Runs a unit test for coverage in a subprocess.

  The 'def' statements of a file are only tested upon import,
  so if we run multiple tests in the same process, all tests
  after the first fail their coverage requirement, because the
  'def' statements all got run during the setup for the first test.

  Running each test in a subprocess allows each test to do its
  own import.
  '''

  # SCons for some reason imports cPickle and replaces the built-in
  # pickle with it, which Process chokes on when trying to start
  # the sub-process.  Restore the regular pickle.

  cPickleModule = sys.modules['pickle']
  moduleName = 'pickle'
  pickleModule = importlib.import_module(moduleName)
  sys.modules['pickle'] = pickleModule

  # Force MacOS to use fork, rather than spawn.
  fork = multiprocessing.get_context('fork')

  queue: Queue = Queue()
  process = fork.Process(target=runCovTestQueue,
                         args=(target, source, env, queue))

  process.start()
  process.join()
  exitCode = process.exitcode

  # exitCode is only None when the process hasn't finished yet.
  assert exitCode is not None

  if exitCode == 0:
    returned = queue.get()
  else:
    returned = exitCode

  # Replace pickle with cPickle as SCons likes.
  sys.modules['pickle'] = cPickleModule

  return returned

#-------------------------------------------------------------------
def runCovTestQueue(target: List[Node],
                    source: List[Node],
                    env: SconsEnvironment,
                    queue: Queue) -> None:
  '''Run the runCovTest() function, returning its value in a Queue.
  '''

  queue.put(runCovTest(target, source, env))

#-------------------------------------------------------------------
def runCovTest(target: List[Node],
               source: List[Node],
               env: SconsEnvironment) -> int:
  """Run a unit test for coverage.
  """

  assert len(source) == 1
  assert len(target) == 1

  srcFile = str(source[0])
  assert srcFile.startswith('src/python/')

  targetFile = str(target[0])
  assert targetFile.endswith('.cov-passed')

  srcFileOnly = os.path.split(srcFile)[1]
  testedFileOnly = srcFileOnly[len('Test'):]
  testedFileDir = os.path.split(os.path.split(srcFile)[0])[0]
  testedFile = os.path.join(testedFileDir, testedFileOnly)

  covDataFileBase = os.path.splitext(targetFile)[0]
  covReportDir = covDataFileBase + '-report'
  covReportFile = testedFile.replace('/', '_').replace('.py', '_py.html')

  testedModule = testedFile[len('src/python/'):-len('.py')]
  testedModule = testedModule.replace('/', '.')

  cov = coverage.Coverage(data_file=covDataFileBase,
                          data_suffix='cov-data',
                          source=[testedModule])

  # We must start recording coverage before the tested class
  # is imported, or the 'def' lines will show as uncovered.
  cov.start()

  suite = buildTestSuite(srcFile, env)
  runner = unittest.TextTestRunner(verbosity=2)

  result = runner.run(suite)

  cov.stop()
  cov.save()

  numProblems: int = len(result.errors) + len(result.failures)

  # Return non-zero to make the build fail.
  if numProblems:
    return numProblems

  percentCovered = cov.html_report(directory=covReportDir)

  # Trim to 2 decimal points.
  percentCovered = float("%.2f" % percentCovered)

  print(str(percentCovered) + '% covered')
  print()
  print('  ------------------------------------------')
  print('  -  To see the coverage report, run:')
  print('  -')
  print('  -    google-chrome ' + covReportDir + '/' + covReportFile)
  print('  ------------------------------------------')

  testedClass = testedFile[len('src/python/'):-len('.py')]
  testedClass = testedClass.replace('/', '.')

  minCovered = ai.axe.build.sconstruct.Coverage.minCoverage(testedClass)

  if percentCovered < minCovered:
    print()
    print('* Error.  Class ' + testedClass + ' has only ' +
          str(percentCovered) + '% coverage,')
    print('* but it needs ' + str(minCovered) + '%.')

    return 1

  return 0

#-------------------------------------------------------------------
def buildTestSuite(srcFile: str, env: SconsEnvironment) -> TestSuite:
  '''Builds a test runner that will run the tests in the specified file.
  '''
  assert (srcFile.startswith('src/python/') or
          srcFile.startswith('src/browserTest/'))
  assert srcFile.endswith('.py')

  isBrowserTest = srcFile.startswith('src/browserTest/')

  if isBrowserTest:
    prefix = 'src/browserTest/'
  else:
    prefix = 'src/python/'

  testClassName = srcFile[len(prefix):]

  testClassName = testClassName[:-len('.py')]
  testClassName = testClassName.replace('/', '.')

  testClassNameOnly = testClassName.split('.')[-1]

  testModule = __import__(testClassName, fromlist=[testClassNameOnly])
  testClass = getattr(testModule, testClassNameOnly)

  if isBrowserTest:
    testClass.setIsHeadless(env['isHeadless'])
    testClass.setUrl(env['browserTestUrl'])

  suite = unittest.TestLoader().loadTestsFromTestCase(testClass)
  return suite
