#-------------------------------------------------------------------
#  ModTar.py
#
#  The ModTar module.
#
#  Copyright 2013  Applied Invention, LLC.
#-------------------------------------------------------------------

"""An SCons builder that builds from an existing .tar file.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
from SCons.Node import NodeList
from typing import List
from typing import Tuple
from typing import Union
import os.path
import SCons
from ai.axe.build.sconstruct.scons import Defaults
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def modTarCmd(env: SCons.Environment,
              newTarFile: str,
              oldTarFile: str,
              workDir: str,
              tarFiles: List[Tuple[str, str]]) -> NodeList:
  '''Returns a command that will add files to a .tar.gz.

     @param newTarFile The file you want to create.
     @param oldTarFile The existing file you want to add to.
     @param workDir The directory that the tar file should be unzipped into.
     @param tarFiles A list of (fileName, destPath) tuples.
  '''

  # The directory that untarring the old tar file will create.
  tarDirBase = os.path.split(oldTarFile)[1]
  assert tarDirBase.endswith(".tar")
  tarDirBase = tarDirBase[:-len(".tar")]

  tarDir = os.path.join(workDir, tarDirBase)

  srcFiles = [oldTarFile]
  actions = []

  # Untar the source tar.

  actions.append(Defaults.Delete(workDir))
  actions.append(Defaults.Mkdir(workDir))
  actions.append(env.Action("gtar xf " + oldTarFile + " -C " + workDir))
  actions.append(Defaults.Mkdir(os.path.join(tarDir, "conf")))

  # Add the files.

  for fileName, destPath in tarFiles:
    src = fileName
    dest = os.path.join(tarDir, destPath, os.path.split(fileName)[1])
    if '*' in src:
      # For a wildcard copy, the dest is the directory only.
      dest = os.path.join(tarDir, destPath)
    actions.extend(copyActions(dest, src))
    if '*' in src:
      # For a wildcard copy, use the parent dir as a dependency.
      src = os.path.split(src)[0]
    srcFiles.append(src)

  # Tar the files.
  parentDir, dirOnly = os.path.split(tarDir)
  tarCmd = "gtar cf " + newTarFile + " -C " + parentDir + " " + dirOnly
  actions.append(env.Action(tarCmd, "Creating tar file " + newTarFile + "..."))
  tar = env.Command(newTarFile, srcFiles, actions)

  env.Clean(tar, workDir)

  return tar

#-------------------------------------------------------------------
def copyActions(dest: str, src: str) -> List[Union[str, SCons.Action]]:

  if os.path.isdir(src):

    # Don't include the copied directory name in the destination, to avoid
    # creating a nested directory when the directory already existed.
    if os.path.split(src)[1] == os.path.split(dest)[1]:
      dest = os.path.split(dest)[0]

    return ["cp -rf " + src + " " + dest]

  elif '*' in src:

    # A wildcard copy.
    return ["mkdir -p " + dest, "cp -rf " + src + " " + dest]

  else:
    return [Defaults.Copy(dest, src)] # type: ignore
