#-------------------------------------------------------------------
#  util.py
#
#  The util module.
#
#  Copyright 2013  Applied Invention, LLC.
#-------------------------------------------------------------------

"""Utility functions used by the build system.
"""

#-------------------------------------------------------------------
# Import statements go here.
#
from glob import glob
from typing import List
import os.path
import fnmatch
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
def recursiveGlob(rootDir: str, pattern: str) -> List[str]:
  '''Look for all files whose name match the pattern, starting in rootDir.
  '''

  found: List[str] = []
  for dirpath, unusedDirnames, filenames in os.walk(rootDir):
    for f in fnmatch.filter(filenames, pattern):
      found.append(os.path.join(dirpath, f))

  found.sort()
  return found

#-------------------------------------------------------------------
def globEx(pattern: str, maxDepth: int = 10) -> List[str]:
  '''Like glob(), except a single ** is allowed.

  @param pattern A glob()-compatible string, with a single ** allowed.
                 The ** sequence means will match any directories.
  @param maxDepth How many directories the ** can match.

  @return A list of file name strings.
  '''

  if pattern.count('**') > 1:
    msg = "Only 1 ** is allowed per pattern.  Pattern:  " + pattern
    raise Exception(msg)

  files: List[str] = []
  for i in range(maxDepth):
    newPattern = pattern.replace("**", "/".join(["*"] * i))
    files.extend(glob(newPattern))

  files.sort()
  return files

#-------------------------------------------------------------------
def fileExists(fileName: str) -> bool:
  '''Returns true if the file exists.

  MacOS has a case-insensitive file system, so os.path.exist() will
  return True even if the case is wrong.  This function is case-sensitive.

  @param fileName The file to check for.
  '''

  path, fileOnly = os.path.split(fileName)
  if not path:
    path = '.'

  return fileOnly in os.listdir(path) and os.path.isfile(fileName)

#-------------------------------------------------------------------
def isInPath(programName: str) -> bool:
  '''Returns true if the specified program is found on the path.

  We used to use 'which foo' and check the exit value, but on Solaris
  'which' always exits with zero whether the program is found or not.'''

  for path in os.environ["PATH"].split(os.pathsep):
    candidate = os.path.join(path, programName)
    if os.path.isfile(candidate) and os.access(candidate, os.X_OK):
      return True

  return False
