#-------------------------------------------------------------------
#  Construct.py
#
#  The Construct class.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the Construct class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .scons.Util import silent
from SCons.Environment import Base as SconsEnvironment
from SCons.Node import Node
from SCons.Script import ARGUMENTS
#from SCons.Script import COMMAND_LINE_TARGETS
from SCons.Script import Environment
from ai.axe.build import BuildSetup
from ai.axe.build import BuildIdFile
from ai.axe.build import VersionFile
from ai.axe.build.sconstruct.Util import fileExists
from ai.axe.build.sconstruct.Util import recursiveGlob
#from ai.axe.build.sconstruct.scons.Targets import Targets
from ai.axe.build.sconstruct.scons.Util import newSectionProgress
from ai.axe.util import StringUtil
from collections import OrderedDict
from typing import Any
from typing import Dict
from typing import List
from typing import Tuple
import ai.axe.build.sconstruct.Coverage
import ai.axe.build.sconstruct.scons.Copy
import ai.axe.build.sconstruct.scons.Defaults
import ai.axe.build.sconstruct.scons.PyTest
import ai.axe.build.sconstruct.scons.TarGz
import ai.axe.build.sconstruct.scons.TsLint
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class Construct:
  '''An SCons construction setup.
  '''

  #-----------------------------------------------------------------
  def __init__(self,
               buildSetup: BuildSetup) -> None:
    '''Creates a new Construct object.

    @param versionName The version that we are building.
                       Will be an empty string if no version name was specified
                       by the user.
    '''

    # The version passed in on the command line.
    versionName = ARGUMENTS.get('version', '')

    # The name of the app being built.
    self.appName = buildSetup.appName()

    # The name of the app being built.
    self.appNameCap = StringUtil.capitalize(self.appName)

    # The name of the app being built.
    self.appNameUpper = StringUtil.camelCaseToUnderscores(self.appName).upper()

    # Directory where build tools are located.
    self.toolDir: str = "not set yet"

    # The SCons Environment.
    self.env: SconsEnvironment = Environment()

    # The version that we are building.
    self.versionFile: VersionFile = VersionFile(versionName)

    # The build ID that we are building.
    self.buildIdFile: BuildIdFile = BuildIdFile()

    # The string name of the root Python package.
    self.rootPackage: str = buildSetup.rootPackageName()

    # The directory where the root package is located.
    self.rootPackageDir = 'src/python/' + self.rootPackage.replace('.', '/')

    # The list of available targets, for display to the user.
    self.targets: Dict[str, str] = OrderedDict()

    # The 'scons targets' text to display to the user.
    self.targetsHeader: str = '''\

Scons targets
-------------

You run these by listing them after 'scons'.  For example, to run the unit
tests, you enter 'scons test' on the command line.

The following are the main targets.  To see all the targets, run
scons allTargets.

'''

    # The major targets to be included in the 'scons targets' text.
    self.majorTargets: List[Tuple[str, str]] = [
      ('test', 'Run all the unit tests.'),
      ('test-CLASS_NAME', 'Run the unit test for the specified class.'),
      ('cover', 'Run all the unit tests, checking for coverage.'),
      ('cover-CLASS_NAME', 'Check coverage for the specified class.'),
      ('pylint', 'Lint all python files.'),
      ('pylint-PACKAGE', 'Lint all files in the specified package.'),
    ]

  #-----------------------------------------------------------------
  def addTarget(self,
                name: str,
                description: str,
                isMajor: bool = False) -> None:
    '''Adds the specfied target to the list displayed to the user.

    @param isMajor Whether this is a major target that should be added to the
                   'targetsText' string.
    '''

    self.targets[name] = description

    if isMajor:
      self.majorTargets.append((name, description))

  #-----------------------------------------------------------------
  def printTargets(self) -> None:
    '''Prints the short targets text.
    '''

    print(self.targetsHeader)

    nameWidth = max((len(x[0]) for x in self.majorTargets))
    descriptionWidth = max((len(x[1]) for x in self.majorTargets))

    formatStr = '  %-' + str(nameWidth) + 's  %-' + str(descriptionWidth) + 's'

    for name, description in self.majorTargets:
      print(formatStr % (name, description))

  #-----------------------------------------------------------------
  def printAllTargets(self) -> None:
    '''Prints the full list of targets.
    '''

    allTargetsText = '\n\nAll Scons targets, in alphabetical order.\n\n'

    allTargetsList = sorted(self.targets.keys())
    for targetName in allTargetsList:
      allTargetsText += targetName + '\n'
      allTargetsText += '    ' + self.targets[targetName] + '\n'

    print(allTargetsText)

  #-----------------------------------------------------------------
  def printAllTargetNames(self) -> None:
    '''Prints the full list of target names.

    This is intended for shell tab autocompletion.
    '''

    allTargetsList = sorted(self.targets.keys())
    allTargetsText = '\n'.join(allTargetsList)

    print(allTargetsText)

  #-----------------------------------------------------------------
  @staticmethod
  def create(buildSetup: BuildSetup) -> 'Construct':
    '''Run this file to set up SCons to build your file.
    '''

    BuildSetup.register(buildSetup)

    construct = Construct(buildSetup)
    construct.addTargets()

    return construct

  #-----------------------------------------------------------------
  def addTargets(self) -> None:
    '''Add targets to this Construct object.
    '''

    buildSetup = BuildSetup.get()

    appNameCap = self.appNameCap
    appNameUpper = self.appNameUpper
    rootPackage = self.rootPackage
    rootPackageDir = self.rootPackageDir

    #----------------------------------------------------------------------
    def addTarget(name: str, description: str) -> None:
      self.addTarget(name, description)

    # Allow capital variable names in this file.
    # pylint: disable=C0103

    #----------------------------------------------------------------------
    # Set up references to external files, programs, and so on.

    # The SCons environment in which we'll run our commands.
    aenv = self.env

    toolDir = os.environ[appNameUpper + '_TOOLS']
    self.toolDir = toolDir

    # Add the AXE environment variables.
    aenv['ENV']['PYTHONPATH'] = os.environ['PYTHONPATH']
    aenv['ENV'][appNameUpper + '_CONFIG'] = os.environ[appNameUpper + '_CONFIG']

    if 'NODE_PATH' in os.environ:
      aenv['ENV']['NODE_PATH'] = os.environ['NODE_PATH']

    # Look at the contents of a target to see if it needs to be rebuilt.
    aenv.Decider("MD5-timestamp")

    # Cache all implicit depenencies.
    aenv.SetOption("implicit_cache", True)
    aenv.SetOption("max_drift", 1)

    # Add Builders.
    ai.axe.build.sconstruct.scons.TarGz.addGzipBuilder(aenv)
    ai.axe.build.sconstruct.scons.TsLint.addTsLintBuilder(aenv)

    ## Location of the jshint executable.
    #jshintConfigSrc = 'admin/js/jshintrc'
    #jshintConfig = 'build/jslint/jshintrc'
    #jshint = (toolDir + '/bin/node ' + toolDir + '/bin/jshint --config ' +
    #          jshintConfig + ' --reporter admin/js/jsHintReporter.js')

    # Mypy
    mypy = toolDir + '/bin/python3 ' + toolDir + '/bin/mypy'
    aenv['MYPY'] = mypy

    # If there's a development version of AXE being used by another project,
    # this will be set.
    axeDevHome = os.environ.get('AXE_DEV_HOME', None)

    #----------------------------------------------------------------------
    # Put arguments passed by the user into variables.

    # The targets that the user specified on the command line.
    #userTargets = Targets(COMMAND_LINE_TARGETS)

    #----------------------------------------------------------------------
    # Rules to build the version .py file.

    newSectionProgress('py version')

    versionFile = self.versionFile

    # If the user is setting a version, delete any existing version file
    # so a new one will get created.
    if versionFile.userVersionName and os.path.isfile(versionFile.fileName()):
      os.remove(versionFile.fileName())

    # Closure that writes the version file.
    def writeFile(target: List[Node],
                  source: List[Node],
                  env: SconsEnvironment) -> None:
      versionFile.writeFile()

      # Supress unused variable warnings.
      # pylint: disable=self-assigning-variable
      target = target
      source = source
      env = env

    action = aenv.Action(writeFile, versionFile.statusMessage())

    aenv.Command(versionFile.fileName(), '', action)

    #----------------------------------------------------------------------
    # Rules to build the Build ID .py file.

    newSectionProgress('build ID')

    buildIdFile = self.buildIdFile

    # Delete any existing file so a new one will get created.
    if os.path.isfile(buildIdFile.fileName()):
      os.remove(buildIdFile.fileName())

    # Closure that writes the build ID file.
    def writeBuildIdFile(target: List[Node],
                         source: List[Node],
                         env: SconsEnvironment) -> None:
      buildIdFile.writeFile()

      # Supress unused variable warnings.
      # pylint: disable=self-assigning-variable
      target = target
      source = source
      env = env

    action = aenv.Action(writeBuildIdFile, buildIdFile.statusMessage())

    aenv.Command(buildIdFile.fileName(), '', action)

    #----------------------------------------------------------------------
    # Rules to run the unit tests.

    newSectionProgress('test')

    allTestedClassNames = []

    ai.axe.build.sconstruct.scons.PyTest.addPyTestBuilder(aenv)

    testSrcFiles = recursiveGlob(rootPackageDir, 'Test*.py')
    testSrcFiles.sort()

    for testSrcFile in testSrcFiles:

      # Make the path relative to the build directory.
      relName = testSrcFile[len(rootPackageDir + '/'):]
      testedClassName = relName.split('/')[-1]
      testedClassName = testedClassName[len('Test'):-len('.py')]

      testedPackageName = rootPackage + '.'
      testedPackageName += '.'.join(relName.split('/')[:-2])

      # Check to see whether it's a class or module.
      testedClassDir = os.path.dirname(os.path.dirname(testSrcFile))
      testedModuleName = testedClassName[0].lower() + testedClassName[1:]
      if fileExists(os.path.join(testedClassDir, testedModuleName) + '.py'):
        testedClassName = testedModuleName

      testPassedName = 'build/unittest/' + relName
      testPassedName = testPassedName[:-len('.py')]
      testPassedName += '.passed'

      covPassedName = 'build/coverage/' + relName
      covPassedName = covPassedName[:-len('.py')]
      covPassedName = os.path.split(os.path.split(covPassedName)[0])[0]
      covPassedName = os.path.join(covPassedName, testedClassName)
      covPassedName += '.cov-passed'

      # Someday, do real dependency tracking.
      # For now, just consider the class test file as the input
      # whose change will force a re-run.

      # pylint: disable=E1101
      aenv.PyTest(testPassedName, testSrcFile)
      aenv.PyCov(covPassedName, testSrcFile)
      # pylint: enable=E1101

      aenv.Alias('test', testPassedName)
      aenv.Alias('cover', covPassedName)

      addTarget('test', 'Run all unit tests.')
      addTarget('cover', 'Run all unit tests, checking for coverage.')


      aenv.Alias('test-' + testedClassName, testPassedName)
      aenv.Alias('cover-' + testedClassName, covPassedName)

      addTarget('test-' + testedClassName,
                'Run the unit test for class ' + testedClassName)
      addTarget('cover-' + testedClassName,
                'Check test coverage for class ' + testedClassName)

      aenv.Alias('test-' + testedPackageName, testPassedName)
      aenv.Alias('cover-' + testedPackageName, covPassedName)

      addTarget('test-' + testedPackageName,
                'Run all unit tests in package ' + testedPackageName)
      addTarget('cover-' + testedPackageName,
                'Check coverage for classes in package ' + testedPackageName)

      # Until dependencies are working, always re-run unit tests.
      aenv.AlwaysBuild(testPassedName)

      allTestedClassNames.append((testedPackageName, testedClassName))

    #----------------------------------------------------------------------
    # Rules to find untested files.

    newSectionProgress('untested')

    srcFiles = recursiveGlob(rootPackageDir, '*.py')
    srcFiles.sort()

    for srcFile in srcFiles[:]:

      if '/test/' in srcFile:
        srcFiles.remove(srcFile)

      elif '/actions/' in srcFile:
        srcFiles.remove(srcFile)

      elif srcFile.endswith('__init__.py'):
        srcFiles.remove(srcFile)

      elif srcFile.endswith(appNameCap + 'Version.py'):
        srcFiles.remove(srcFile)

    testedClassNames = [nameTuple[1] for nameTuple in allTestedClassNames]

    untestedFiles = []
    for srcFile in srcFiles:

      srcClassName = srcFile.split('/')[-1]
      srcClassName = srcClassName[:-len('.py')]

      if (srcClassName not in testedClassNames and
          srcFile not in buildSetup.coverageExemptions()):
        untestedFiles.append(srcFile)

    def writeUntestedFiles(target: List[Node],
                           source: List[Node],
                           env: SconsEnvironment) -> int:

      # Suppress unused variable warning.
      # pylint: disable=self-assigning-variable
      source = source
      env = env

      targetName = str(target[0])
      open(targetName, 'w', encoding='utf-8').write('\n'.join(untestedFiles))

      if untestedFiles:
        print('The following files are missing unit tests:')
        print()
        print('\n'.join(untestedFiles))

      return len(untestedFiles)

    msg = 'Checking for untested files...'
    action = aenv.Action(writeUntestedFiles, msg)
    actionSrc = srcFiles[:]
    cmd = aenv.Command('build/unittest/untested.txt', actionSrc, action)

    aenv.Alias('untested', cmd)
    addTarget('untested', 'Find classes that are missing a unit test.')

    def writeTestedFiles(target: List[Node],
                         source: List[Node],
                         env: SconsEnvironment) -> int:

      # Suppress unused variable warning.
      # pylint: disable=self-assigning-variable
      source = source
      env = env

      targetName = str(target[0])
      outFile = open(targetName, 'w')

      for pkg, className in allTestedClassNames:
        outFile.write(pkg + ' ' + className + '\n')

      return 0

    msg = 'Writing tested files...'
    action = aenv.Action(writeTestedFiles, msg)
    actionSrc = srcFiles[:]
    cmd = aenv.Command('build/unittest/tested.txt', actionSrc, action)

    aenv.Alias('tested', cmd)

    def writeNonexistentFiles(target: List[Node],
                              source: List[Node],
                              env: SconsEnvironment) -> int:

      # Suppress unused variable warning.
      # pylint: disable=self-assigning-variable
      source = source
      env = env

      exemptions = buildSetup.coverageExemptions()
      nonexistentFiles = [x for x in exemptions if not os.path.exists(x)]

      targetName = str(target[0])
      open(targetName, 'w', encoding='utf-8').write('\n'.join(nonexistentFiles))

      if nonexistentFiles:
        msg = ('The following files are listed in the coverage exemptions ' +
               "list, but don't exist:\n" +
               '\n' +
               '\n'.join(nonexistentFiles))
        print(msg)

      return len(untestedFiles)

    msg = 'Writing bad coverage exemption rule files...'
    action = aenv.Action(writeNonexistentFiles, msg)
    actionSrc = srcFiles[:]
    cmd = aenv.Command('build/unittest/nonexistentExemptions.txt',
                       actionSrc, action)

    aenv.Alias('nonexistentExemptions', cmd)

    #----------------------------------------------------------------------
    # Rules to lint the Python code.

    newSectionProgress('pylint')

    allDirs = [x[0] for x in os.walk(rootPackageDir)]
    allDirs = [x for x in allDirs if not x.endswith('__pycache__')]
    allDirs.sort()

    class PyLintRunner:

      def __init__(self, pkgName: str) -> None:
        self.pkgName: str = pkgName

      def __call__(self,
                   target: List[Node],
                   source: List[Node],
                   env: SconsEnvironment) -> None:

        # pylint: disable=import-outside-toplevel
        import pylint.lint

        args = [self.pkgName]
        pylint.lint.Run(args)

    for pkgDir in allDirs:

      assert pkgDir.startswith('src/python/')
      pkgName = pkgDir[len('src/python/'):]
      pkgName = pkgName.replace('/', '.')

      outFile = 'build/pylint/' + pkgName + '.passed'

      # Also check the Axe source files if we've got an AXE_DEV_HOME.
      myPyAxeDevRoot = (' ' + axeDevHome + '/src/python') if axeDevHome else ''

      # Create the list of paths to for mypy to search for stubs and classes.

      myPyPaths: List[str] = []

      # Add any stubs provided by this project.
      myPyPaths.append('src/pystubs')

      if axeDevHome:

        # Add Axe-supplied stubs (i.e. the Axe sqlalchemy stubs).
        myPyPaths.append(axeDevHome + '/src/pystubs')

        # Don't need to add Axe classes to the path because we're
        # adding them to the list of classes to check (myPyAxeDevRoot)

      else:
        aiFileLocation = os.path.split(os.path.split(ai.__file__)[0])[0]

        # Add Axe class type hints.
        myPyPaths.append(aiFileLocation)

        # Add Axe-supplied stubs (i.e. the Axe sqlalchemy stubs).
        myPyPaths.append(aiFileLocation + '/pystubs')

      myPyEnv = 'MYPYPATH=' + ':'.join(myPyPaths)

      cmd = myPyEnv + " $MYPY " + pkgDir + myPyAxeDevRoot
      msg = 'Type-checking package %s...' % pkgName
      action1 = aenv.Action(cmd, msg)

      cmds = [PyLintRunner(pkgName),
              'touch ' + outFile]
      msg = 'Linting package %s...' % pkgName
      action2 = aenv.Action(cmds, msg)

      cmd = aenv.Command(outFile, pkgDir, [action1, action2])

      aenv.Alias('pylint-' + pkgName, cmd)
      addTarget('pylint-' + pkgName,
                'Lint all the python classes in package ' + pkgName)

    aenv.Alias('pylint', 'build/pylint/' + rootPackage + '.passed')
    addTarget('pylint', 'Lint all the python classes.')

    #----------------------------------------------------------------------
    # The targets pseudo-target to list all targets.
    newSectionProgress('targets')


    # pylint: disable=W0613

    def showTargets(env: Any, target: Any, source: Any) -> None:
      self.printTargets()

    def showAllTargets(env: Any, target: Any, source: Any) -> None:
      self.printAllTargets()

    def showAllTargetNames(env: Any, target: Any, source: Any) -> None:
      self.printAllTargetNames()

    # pylint: enable=W0613

    action = aenv.Action(showTargets, silent)
    targets = aenv.Command('targets', None, action)

    aenv.Alias('help', targets)
    aenv.Alias('targets', targets)
    aenv.AlwaysBuild(targets)

    action = aenv.Action(showAllTargets, silent)
    allTargets = aenv.Command('allTargets', None, action)

    aenv.Alias('allTargets', allTargets)
    aenv.AlwaysBuild(allTargets)

    action = aenv.Action(showAllTargetNames, silent)
    allTargetNames = aenv.Command('allTargetNames', None, action)

    aenv.Alias('allTargetNames', allTargetNames)
    aenv.AlwaysBuild(allTargetNames)

    #----------------------------------------------------------------------
    # Default is to show the list of targets.

    aenv.Default("targets")

    # This will make the
    # default is to build everything and install
    #aenv.Default(".")
