#-------------------------------------------------------------------
#  compress.py
#
#  The compress module.
#
#  Copyright 2013 Applied Invention, LLC.
#-------------------------------------------------------------------

'''Compress multiple JS/CSS/HTML files into one.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.js.jsSourceMap import LineColumn
from ai.axe.js.jsSourceMap import SourceMapGenerator
from ai.axe.js import ParseJsComment
from typing import List
from typing import Match
from typing import Tuple
import json
import os.path
import re
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
def genDebugHtml(srcRoot: str,
                 srcHtml: str,
                 jsListFileName: str,
                 extJsListFileName: str) -> str:
  '''Generate a debug HTML file.

  @param srcRoot The root directory of the source tree.
  @param srcHtml The HTML file to modify.
  @param ramJsFileName The name of the RAM JS file to include.
  @param extJsFileName The name of the external JS file to include.

  @return The text of the HTML file.
  '''

  text = open(os.path.join(srcRoot, srcHtml), encoding='utf-8').read()

  elemText = genJsElements(srcRoot, jsListFileName)
  stubText = '<!-- Build system inserts RAM JS here. -->'
  text = text.replace(stubText, elemText)

  elemText = genJsElements(srcRoot, extJsListFileName)
  stubText = '<!-- Build system inserts 3rd-party JS here. -->'
  text = text.replace(stubText, elemText)

  return text

#-------------------------------------------------------------------
def genReleaseHtml(srcRoot: str,
                   srcHtml: str,
                   jsListFileName: str,
                   extJsListFileName: str) -> str:
  '''Generate a release HTML file.

  @param srcRoot The root directory of the source tree.
  @param srcHtml The HTML file to modify.
  @param jsListFileName The name of the RAM JS file to include.
  @param extJsListFileName The name of the external JS file to include.

  @return The text of the HTML file.
  '''

  replaceTexts = (
    ('<!-- Build system inserts RAM JS here. -->',
     '<script src="' + jsListFileName + '"></script>'),

    ('<!-- Build system inserts 3rd-party JS here. -->',
     '<script src="' + extJsListFileName + '"></script>'),

    ('<link href="ramIncludes.css" rel="stylesheet" />',
     '<link href="ram.css" rel="stylesheet" />'),
  )

  text = open(os.path.join(srcRoot, srcHtml), encoding='utf-8').read()

  for oldText, newText in replaceTexts:
    text = text.replace(oldText, newText)

  return text

#-------------------------------------------------------------------
def genJsElements(srcRoot: str,
                  jsListFileName: str,
                  indent: int = 4) -> str:
  '''Generate source tags for the files listed in the specified source file.

  @param srcRoot The root directory of the source tree.
  @param jsListFileName The name of a file containing a list of JS files.
  @param indent How many spaces to indent the returned text.

  @return A string that is a script tag per file name.
  '''

  listFile = os.path.join(srcRoot, jsListFileName)
  fileNames = readJsSourceListFile(listFile)

  tagTexts = []

  for fileName in fileNames:

    tagText = '<script src="' + fileName + '"></script>'
    tagTexts.append(tagText)

  indentText = ' ' * indent

  return indentText + ('\n' + indentText).join(tagTexts)

#-------------------------------------------------------------------
def genJsFile(srcRoot: str,
              jsListFileName: str,
              genFileName: str,
              extJsRoot: str) -> Tuple[str, str, List[str]]:
  '''Compress the files listed in the specified source file into a single file.

  @param jsListFileName The name of a file containing a list of JS files.
  @param srcRoot The root directory of the source tree.
  @param genFileName The name of the file being generated.
  @param extJsRoot The root directory to load JS files from.

  @return A tuple: (1) the contents of the files concatenated,
          (2) a source map, and (3) a list of all the source files.
  '''

  listFile = os.path.join(srcRoot, jsListFileName)
  fileNames = readJsSourceListFile(listFile)

  fileTexts = []
  srcMap = SourceMapGenerator(genFileName)

  # Zero-based current line number.
  currentLineNum = 0

  for fileName in fileNames:

    if fileName.startswith('jslib/'):
      # Trim the leading jslib.
      if extJsRoot.endswith('jslib-app'):
        fileName = fileName[len('jslib/'):]

      filePath = extJsRoot
    else:
      filePath = srcRoot

    fullName = os.path.join(filePath, fileName)
    if not os.path.exists(fullName):
      msg = "Error processing JS include file: " + listFile + "\n"
      msg += "It contains an include for non-existent file '" + fullName + "'."
      raise Exception(msg)
    text = open(fullName, 'r', encoding='utf-8').read()

    # Our comment parser can't handle ext.js, so only minify our code.
    # ext.js is already minified anyway.
    if genFileName != 'ext.js':
      text = blankJsComments(text)

    text, htmlFileNames = insertHtmlIntoJs(srcRoot, text)
    text += '\n'
    fileTexts.append('// Source: ' + fileName + '\n')
    fileTexts.append(text)

    # Add the mappings between the source lines and generated lines.

    currentLineNum += 1
    for lineNum in range(0, text.count('\n')):

      # One-based line numbers.
      genLineNum = currentLineNum + 1
      srcLineNum = lineNum + 1

      mapping = {
        'generated': LineColumn(genLineNum, 0),
        'original': LineColumn(srcLineNum, 0),
        'source': fileName
      }

      srcMap.addMapping(**mapping)

      currentLineNum += 1

  fileContents = u''.join(fileTexts)
  fileContents += '//# sourceMappingURL=' + genFileName + '.map'

  allFileNames = fileNames[:] + htmlFileNames

  sourceMapText = json.dumps(srcMap.toJson())
  return fileContents, sourceMapText, allFileNames

#-------------------------------------------------------------------
def readJsSourceListFile(jsListFileName: str) -> List[str]:
  '''Reads a JS source list file, and returns a list of file names.
  '''

  text = open(jsListFileName, 'r', encoding='utf-8').read()

  # Remove comments.
  text = re.sub(r'#.*$', '', text, flags=re.MULTILINE)

  fileNames = []

  for line in text.split('\n'):
    line = line.strip()
    if line:
      fileNames.append(line)

  return fileNames

#-------------------------------------------------------------------
def insertHtmlIntoJs(srcRoot: str, text: str) -> Tuple[str, List[str]]:
  '''Make a reference to an HTML file a string of inline HTML text.
  '''

  pat = r"'templateUrl' : '([^']*)'"
  replacer = HtmlInserter(srcRoot)

  text = re.sub(pat, replacer, text)

  return text, replacer.htmlFileNames

#-------------------------------------------------------------------
class HtmlInserter:
  '''Make a reference to an HTML file a string of inline HTML text.
  '''

  def __init__(self, srcRoot: str) -> None:
    self.srcRoot: str = srcRoot
    self.htmlFileNames: List[str] = []

  def __call__(self, match: Match) -> str:

    fileName = match.group(1)
    self.htmlFileNames.append(fileName)

    fileText = open(os.path.join(self.srcRoot, fileName),
                    encoding='utf-8').read()

    # Convert to a single line.
    fileText = fileText.replace('\r', '')
    fileText = fileText.replace('\n', ' ')

    # Escape quotes.
    fileText = fileText.replace("'", r"\'")

    return "'template' : '" + fileText + "'"

#-------------------------------------------------------------------
def blankJsComments(text: str) -> str:
  '''Removes any comments in the specified JS text.
  '''

  commentBeginEnds = ParseJsComment.findComments(text)

  if not commentBeginEnds:
    return text

  newText = text[:commentBeginEnds[0][0]]

  for i, (begin, end) in enumerate(commentBeginEnds):
    if i > 0:
      unusedPrevBegin, prevEnd = commentBeginEnds[i - 1]
      newText += text[prevEnd:begin]

    comment = text[begin:end]
    numLines = comment.count('\n')
    newText += ('\n' * numLines)

  newText += text[commentBeginEnds[-1][-1]:]

  return newText

#-------------------------------------------------------------------
cssCommentRe = re.compile(r'/\*.+?\*/', re.DOTALL)

#-------------------------------------------------------------------
def blankCssComments(text: str) -> str:
  '''Removes any comments in the specified CSS text.
  '''

  return cssCommentRe.sub('', text)

#-------------------------------------------------------------------
def genCssFile(srcRoot: str,
               cssListFileName: str,
               genFileName: str) -> Tuple[str, str, List[str]]:
  '''Compress the files listed in the specified source file into a single file.

  @param cssListFileName The name of a file containing a list of CSS files.
  @param srcRoot The root directory of the source tree.
  @param genFileName The name of the file being generated.

  @return A tuple: (1) the contents of the files concatenated,
          (2) a source map, and (3) a list of source file names.
  '''

  listFile = os.path.join(srcRoot, cssListFileName)
  fileNames = readCssSourceListFile(listFile)

  fileTexts = []
  srcMap = SourceMapGenerator(genFileName)

  # Zero-based current line number.
  currentLineNum = 0

  for fileName in fileNames:

    fullName = os.path.join(srcRoot, fileName)
    if not os.path.exists(fullName):
      msg = "Error processing CSS include file: " + listFile + "\n"
      msg += "It contains an include for non-existent file '" + fileName + "'."
      raise Exception(msg)

    text = open(fullName, 'r', encoding='utf-8').read()

    text = blankCssComments(text)
    text = fixCssPaths(text)
    text += '\n'
    fileTexts.append('/* Source: ' + fileName + ' */' + '\n')
    fileTexts.append(text)

    # Add the mappings between the source lines and generated lines.

    currentLineNum += 1
    for lineNum in range(0, text.count('\n')):

      # One-based line numbers.
      genLineNum = currentLineNum + 1
      srcLineNum = lineNum + 1

      mapping = {
        'generated': LineColumn(genLineNum, 0),
        'original': LineColumn(srcLineNum, 0),
        'source': fileName
      }
      srcMap.addMapping(**mapping)

      currentLineNum += 1

  fileContents = ''.join(fileTexts)
  fileContents += '/*# sourceMappingURL=' + genFileName + '.map */'

  sourceMapText = json.dumps(srcMap.toJson())
  return fileContents, sourceMapText, fileNames

#-------------------------------------------------------------------
def fixCssPaths(text: str) -> str:
  '''Fixes the root of paths in a block of CSS text.
  '''

  replaceTexts = (
    ("url('../", "url('htmllib/"),
    ('url("../', 'url("htmllib/'),
    ('url(../', 'url(htmllib/'),
  )

  for oldText, newText in replaceTexts:
    text = text.replace(oldText, newText)

  return text

#-------------------------------------------------------------------
def readCssSourceListFile(cssListFileName: str) -> List[str]:
  '''Reads a CSS source list file, and returns a list of file names.
  '''

  linePrefix = "@import url('"
  lineSuffix = "');"

  text = open(cssListFileName, 'r', encoding='utf-8').read()

  fileNames = []

  for line in text.split('\n'):
    line = line.strip()
    if line.startswith(linePrefix) and line.endswith(lineSuffix):
      name = line[len(linePrefix):-len(lineSuffix)]

      fileNames.append(name)

  return fileNames
