#-------------------------------------------------------------------
#  WebAction.py
#
#  The WebAction class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the WebAction class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import os
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class WebAction(NewClassCommand):
  '''Command to create a new web action.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new WebAction.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngDirective'.
    '''

    return "webAction"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "A web action"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass webAction noun [verb1 verb2 verb3...]"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Create a new package directory for the noun, and a module file for each verb.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    noun = args.pop(0)
    verbs = args

    if noun.endswith('.py'):
      noun = noun[:-3]
    noun = noun[0].upper() + noun[1:]
    nounLower = noun[0].lower() + noun[1:]

    for i, verb in enumerate(verbs):
      verbs[i] = verb[0].upper() + verb[1:]

    importStrs = []
    for verb in verbs:
      importStrs.append('from . import ' + verb + noun)
    importsStr = '\n'.join(importStrs)

    names['NOUN'] = noun
    names['NOUN_LOWER'] = nounLower
    names['ALL_VERB_IMPORTS'] = importsStr

    initOutFile = nounLower + '/__init__.py'

    if os.path.exists(nounLower):
      print("Error: directory %s already exists." % (nounLower, ))
      sys.exit(1)

    # Generate the package directory and __init__ file.

    os.mkdir(nounLower)
    NewClassWriter.genFile(WebAction.initTemplate, names, initOutFile)

    # Generate each verb file.

    for verb in verbs:

      names['VERB'] = verb
      names['VERB_LOWER'] = verb[0].lower() + verb[1:]
      outFile = nounLower + '/' + verb + noun + '.py'

      NewClassWriter.genFile(WebAction.template, names, outFile)

    hint = """
Hint:  Don't forget to open up src/python/*/actions/__init__.py and add:

  from . import """ + nounLower + """
"""

    print(hint)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)

  #----------------------------------------------------------------
  template = """\
#-------------------------------------------------------------------
#  ${VERB}${NOUN}.py
#
#  The ${VERB}${NOUN} web action module.
#
#  2021 Applied Invention, LLC
#-------------------------------------------------------------------

'''The ${VERB}${NOUN} web action module.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import your class here
from ai.axe.classJson import ClassJsonField
from ai.axe.web.api.desc import JsonResponseDesc
from ai.axe.web.api.desc import WebVerb
from ai.axe.web.api.desc import WebParam
from sqlalchemy.orm.session import Session
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
@WebVerb([WebParam('cropZoneId', int)],
          JsonResponseDesc([ClassJsonField('${NOUN_LOWER}', ${NOUN})]))
def ${VERB_LOWER}${NOUN}(session: Session,
                         userInfo: UserInfo,
                         cropZoneId: int) -> Dict[str, object]:
  '''TODO Add comment here.

  @webParam cropZoneId TODO Add comment here.
  '''TODO Add comment here.

  userInfo.checkCanReadCropZone(session, cropZoneId)

  ${NOUN_LOWER} = ${NOUN}Db.readForCropZone(session, cropZoneId)

  return {"$NOUN_LOWER" : ${NOUN_LOWER}}
"""

  #-------------------------------------------------------------------
  #-------------------------------------------------------------------
  initTemplate = """\
#-------------------------------------------------------------------
#  __init__.py
#
#  Package __init__ file.
#
#  2021 Applied Invention, LLC
#-------------------------------------------------------------------

'''TODO Add comment to describe what the ${NOUN} noun is.
'''TODO Add comment to describe what the ${NOUN} noun is.

#-------------------------------------------------------------------
# Import statements go here.
#
${ALL_VERB_IMPORTS}
#
# Import statements go above this line.
#-------------------------------------------------------------------
"""
