#-------------------------------------------------------------------
#  TsTest.py
#
#  The TsTest class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the TsTest class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class TsTest(NewClassCommand):
  '''Command to create a new Typescript unit test.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new TsTest.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngDirective'.
    '''

    return "tsTest"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "A Typescript unit test"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass tsTest class_name"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Creates a new 'Testclass_name.ts' file with a skeleton unittest in it.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}

    if len(args) > 1:
      print("Error: too many arguments.")
      print(self.usage())
      sys.exit(1)

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    className = args.pop(0)

    if className.endswith('.ts'):
      className = className[:-3]
    if className.startswith('test') or className.startswith('Test'):
      className = className[4:]

    outFile = 'Test' + className + '.ts'
    names['CLASS_NAME'] = className

    NewClassWriter.genFile(TsTest.template, names, outFile)

    hint = """
Hint:  Don't forget to add Test""" + names['CLASS_NAME'] + """ to the
       declarations array in the file:

       src/webapp/htmllib/APP_NAME/APP_NAMEUnitTests.ts
"""

    print(hint)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)

  #----------------------------------------------------------------
  template = """\
//------------------------------------------------------------------
//  Test${CLASS_NAME}.ts
//  2021 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../../axe/util/string';
import { UnitTest } from '../../axe/unittest/UnitTest';
import { UnitTestRunner } from '../../axe/unittest/UnitTestRunner';
import { ${CLASS_NAME} } from '../${CLASS_NAME}';
//------------------------------------------------------------------

/** Unit test for the ${CLASS_NAME} class.
 */
export class Test${CLASS_NAME} extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ${CLASS_NAME} object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  REMOVE_THIS_EXAMPLE_METHOD:

  /** Checks that the foo() method doubles a number.
   */
  testFooDouble() : void
  {
    let doubled: number = Foo.foo(3);

    this.assertEqual("Foo is doubled", 6,  doubled);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class Test${CLASS_NAME}

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new Test${CLASS_NAME}());
"""
