#-------------------------------------------------------------------
#  TsModule.py
#
#  The TsModule class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the TsModule class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class TsModule(NewClassCommand):
  '''Command to create a new Typescript module.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new TsModule.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngDirective'.
    '''

    return "tsModule"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "A Typescript module"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass tsModule module_name"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Creates a new 'module_name.ts' file with a skeleton module in it.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}

    if len(args) > 1:
      print("Error: too many arguments.")
      print(self.usage())
      sys.exit(1)

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    names['MODULE_NAME'] = args.pop(0)
    if names['MODULE_NAME'].endswith('.ts'):
      names['MODULE_NAME'] = names['MODULE_NAME'][:-3]
    outFile = names['MODULE_NAME'] + '.ts'

    NewClassWriter.genFile(TsModule.template, names, outFile)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)

  #----------------------------------------------------------------
  template = """\
//------------------------------------------------------------------
//  ${MODULE_NAME}.ts
//  2021 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../axe/util/string'
//------------------------------------------------------------------

/* BRIEF_DESCIPTION_MODULE_GOES_HERE
 */
BRIEF_DESCIPTION_MODULE_GOES_HERE

//------------------------------------------------------------------
// Constants
//------------------------------------------------------------------

DELETE_THIS_EXAMPLE_CONSTANT

/** The largest integer value allowed in Javascript.
 */
export const MAX_SAFE_INTEGER: number = Math.pow(2, 53) - 1;

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

DELETE_THIS_EXAMPLE_FUNCTION

/** Convert degrees to radians.
 *
 * @param degrees A number of degrees.
 *
 * @return A number of radians.
 */
export function degreesToRadians(degrees: number) : number
{
  return degrees * ram.math.radiansPerDegree;
}
"""
