#-------------------------------------------------------------------
#  TsClass.py
#
#  The TsClass class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the TsClass class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class TsClass(NewClassCommand):
  '''Command to create a new Typescript class.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new TsClass.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngDirective'.
    '''

    return "tsClass"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "A Typescript class"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass tsClass class_name"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Creates a new 'class_name.ts' file with a skeleton class in it.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}

    if len(args) > 1:
      print("Error: too many arguments.")
      print(self.usage())
      sys.exit(1)

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    names['CLASS_NAME'] = args.pop(0)
    if names['CLASS_NAME'].endswith('.ts'):
      names['CLASS_NAME'] = names['CLASS_NAME'][:-3]
    outFile = names['CLASS_NAME'] + '.ts'

    NewClassWriter.genFile(TsClass.template, names, outFile)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)

  #----------------------------------------------------------------
  template = """\
//------------------------------------------------------------------
//  ${CLASS_NAME}.ts
//  2021 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from '../axe/util/string';
//------------------------------------------------------------------

/** Add class documentation here.
 */
export class ${CLASS_NAME}
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  REMOVE_THIS_EXAMPLE_PROPERTY:

  /** The total count.
   */
  count: number = 1;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ${CLASS_NAME} object.
   */
  constructor()
  {
    put_initialization_code_here
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  REMOVE_THIS_EXAMPLE_METHOD:

  /** Adds two numbers.
   *
   * @param x The first number.
   * @param y The second number.
   *
   * @return The sum of the two numbers.
   */
  add(x: number, y: number) : number
  {
    return x + y;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
    ];
    return axeString.formatObject("${CLASS_NAME}", this, propertyNames);
  }

} // END class ${CLASS_NAME}
"""
