#-------------------------------------------------------------------
#  PyPackage.py
#
#  The PyPackage class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the PyPackage class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import os
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class PyPackage(NewClassCommand):
  '''Command to create a new Python package.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new PyPackage.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngDirective'.
    '''

    return "pyPackage"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "A Python package"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass pyPackage package_name"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Creates a new package directory with an __init__.py file in it.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}

    if len(args) > 1:
      print("Error: too many arguments.")
      print(self.usage())
      sys.exit(1)

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    names['PACKAGE_NAME'] = args.pop(0)
    if names['PACKAGE_NAME'].endswith('.py'):
      names['PACKAGE_NAME'] = names['PACKAGE_NAME'][:-3]
    outFile = names['PACKAGE_NAME'] + '/__init__.py'

    if os.path.exists(names['PACKAGE_NAME']):
      print("Error: directory '%s' already exists." % names['PACKAGE_NAME'])
      sys.exit(1)

    os.mkdir(names['PACKAGE_NAME'])

    NewClassWriter.genFile(PyPackage.template, names, outFile)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)

  #----------------------------------------------------------------
  template = """\
#-------------------------------------------------------------------
#  __init__.py
#
#  Package __init__ file.
#
#  2021 Applied Invention, LLC
#-------------------------------------------------------------------

'''The $PACKAGE_NAME package.
'''

#-------------------------------------------------------------------
# Import statements go here.
#

#
# Import statements go above this line.
#-------------------------------------------------------------------
"""
