#-------------------------------------------------------------------
#  PyClass.py
#
#  The PyClass class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the PyClass class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import sys
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class PyClass(NewClassCommand):
  '''Command to create a new Python class.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new PyClass.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngDirective'.
    '''

    return "pyClass"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "A Python class"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass pyClass [-db] class_name"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Creates a new 'class_name.py' file with a skeleton class in it.

If the -db option is provided, a class_name_Db.py DB access module
skeleton will also be created.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}

    dbClass = False
    if '-db' in args:
      dbClass = True
      args.remove('-db')

    if len(args) > 1:
      print("Error: too many arguments.")
      print(self.usage())
      sys.exit(1)

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    names['CLASS_NAME'] = args.pop(0)
    if names['CLASS_NAME'].endswith('.py'):
      names['CLASS_NAME'] = names['CLASS_NAME'][:-3]

    # An initial-lower case class name that can be used as a variable name.
    names['VARIABLE_NAME'] = (names['CLASS_NAME'][0].lower() +
                              names['CLASS_NAME'][1:])

    outFile = names['CLASS_NAME'] + '.py'

    NewClassWriter.genFile(PyClass.template, names, outFile)

    if dbClass:
      outFile = names['CLASS_NAME'] + 'Db.py'
      NewClassWriter.genFile(PyClass.dbTemplate, names, outFile)

    hint = """
Hint:  Don't forget to add:

  from .""" + names['CLASS_NAME'] + """ import """ + names['CLASS_NAME'] + """

to the __init__.py of this package.
"""

    print(hint)

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)

  #----------------------------------------------------------------
  template = """\
#-------------------------------------------------------------------
#  ${CLASS_NAME}.py
#
#  The ${CLASS_NAME} class.
#
#  2021 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the ${CLASS_NAME} class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class ${CLASS_NAME}:
  '''Add class documentation here
  '''Add class documentation here.

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new ${CLASS_NAME}.
    '''

    pass

  #-----------------------------------------------------------------
  def someFunction(self, foo: str, numBar: int) -> str:
    '''One-line description of this function.

    Long description of what this does.

    @exception FooException if there's a problem with the foo.

    @param foo the name of the foo.
    @param numBar how many bars are expected to be returned.

    @return an XML string.
    '''

    if this.x:
      xml += "1"
    else:
      xml += "2"

    return xml

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = []

    return StringUtil.formatRepr(self, attrs)
"""

  #----------------------------------------------------------------
  dbTemplate = """\
#-------------------------------------------------------------------
#  ${CLASS_NAME}Db.py
#
#  The ${CLASS_NAME}Db module.
#
#  2021 Applied Invention, LLC
#-------------------------------------------------------------------

'''Database access for the ${CLASS_NAME} class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from . import ${CLASS_NAME}
from sqlalchemy.orm.session import Session
#
# Import statements go above this line.
#-------------------------------------------------------------------

#-------------------------------------------------------------------
def exists(session: Session, id: int) -> bool:
  '''Checks whether a ${CLASS_NAME} exists.

  @param session The database to read from.
  @param id The ${CLASS_NAME} that will be read.

  @return True if a ${CLASS_NAME} exists, otherwise false.
  '''

  query = session.query(${CLASS_NAME})
  query = query.filter(${CLASS_NAME}.id == id)

  return query.count() > 0

#-------------------------------------------------------------------
def read(session: Session, id: int) -> ${CLASS_NAME}:
  '''Reads the specified ${CLASS_NAME}.

  @param session The database to read from.
  @param id The ${CLASS_NAME} to be read.

  @return A ${CLASS_NAME} object.
  '''

  query = session.query(${CLASS_NAME})
  query = query.filter(${CLASS_NAME}.id == id)

  return query.one()

#-------------------------------------------------------------------
def write(session: Session, ${VARIABLE_NAME}: ${CLASS_NAME}) -> None:
  '''Writes a ${CLASS_NAME} to the databaes.

  @param session The database to write to.
  @param ${VARIABLE_NAME} The ${CLASS_NAME} to write.
  '''

  session.add(${VARIABLE_NAME})
"""
