#-------------------------------------------------------------------
#  NgComponent.py
#
#  The NgComponent class.
#
#  Copyright 2020 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the NgComponent class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.util import StringUtil
from .. import NewClassCommand
from .. import NewClassWriter
from typing import List
import sys
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class NgComponent(NewClassCommand):
  '''Command to create a new Angular component.
  '''

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new NgComponent.
    '''

    NewClassCommand.__init__(self)

  #-----------------------------------------------------------------
  def name(self) -> str:
    '''Returns the name of this command.

    @return A string name, such as 'pyClass' or 'ngComponent'.
    '''

    return "ngComponent"

  #-----------------------------------------------------------------
  def shortDescription(self) -> str:
    '''Returns a short (less than 60 characters) description of this command.

    @return A string description of what this class creates,
            such as 'A Python class.'
    '''

    return "An Angular component"

  #-----------------------------------------------------------------
  def usage(self) -> str:
    '''Returns the usage string for this command.
    '''

    return "Usage:  newClass ngComponent class_name [package_name]"

  #-----------------------------------------------------------------
  def usageDescription(self) -> str:
    '''Returns the usage and description paragraph for this command.
    '''

    return self.usage() + """

Creates a set of js, html, and css files with skeleton code for an AngularJs
component.

If a package_name is supplied, that will be used in the code.  If no
package name is supplied, the current directory will be assumed to be
the package.
"""

  #-----------------------------------------------------------------
  def execute(self, args: List[str]) -> None:
    '''Execute this command.

    @param args The command-line arguments from the user.
    '''

    if '-h' in args or '--help' in args:
      print(self.usageDescription())
      sys.exit(1)

    names = {}
    names['PACKAGE_NAME'] = os.path.basename(os.getcwd())

    if len(args) > 2:
      print("Error: too many arguments.")
      print(self.usage())
      sys.exit(1)

    if not args:
      print("Error: missing argument.")
      print(self.usage())
      sys.exit(1)

    names['CLASS_NAME'] = args.pop(0)
    if len(args) == 1:
      names['PACKAGE_NAME'] = args.pop(0)

    if names['CLASS_NAME'].endswith('.js'):
      names['CLASS_NAME'] = names['CLASS_NAME'][:-3]

    names['VAR_NAME'] = names['CLASS_NAME']

    # Class name is upper case.  Var name is lowercase.
    names['CLASS_NAME'] = self.capitalize(names['CLASS_NAME'])
    names['VAR_NAME'] = self.initialLower(names['VAR_NAME'])

    genFile = NewClassWriter.genFile

    outFile = names['CLASS_NAME'] + '.ts'
    genFile(NgComponent.tsTemplate, names, outFile)

    outFile = names['CLASS_NAME'] + '.html'
    genFile(NgComponent.htmlTemplate, names, outFile)

    outFile = names['CLASS_NAME'] + '.css'
    genFile(NgComponent.cssTemplate, names, outFile)

    hint = """
Hint:  Don't forget to add """ + names['CLASS_NAME'] + """ to the
       declarations array in the file:

       src/webapp/htmllib/APP_NAME/APP_NAMEDeclarations.ts
"""

    print(hint)

  #----------------------------------------------------------------
  def capitalize(self, word: str) -> str:
    return word[0].upper() + word[1:]

  #----------------------------------------------------------------
  def initialLower(self, word: str) -> str:
    return word[0].lower() + word[1:]

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)


  #----------------------------------------------------------------
  #----------------------------------------------------------------
  tsTemplate = """\
//------------------------------------------------------------------
//  ${CLASS_NAME}.ts
//  2021 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
// Import statements go here.
//
import { Component } from '@angular/core';
//
// Import statements go above this line.
//------------------------------------------------------------------

PUT_DESCRIPTION_OF_THE_COMPONENT_HERE
/** PUT_DESCRIPTION_OF_THE_COMPONENT_HERE
 */
@Component({
  'selector': '${CLASS_NAME}',
  'templateUrl' : 'htmllib/${PACKAGE_NAME}/${CLASS_NAME}.html',
  'styleUrls': ['htmllib/${PACKAGE_NAME}/${CLASS_NAME}.css']
})
export class ${CLASS_NAME}
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  REMOVE_THIS_EXAMPLE_PROPERTY:

  /** The total count.
   */
  count: number = 1;

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  /** Initializes a new ${CLASS_NAME} object.
   */
  constructor()
  {
  }

  //----------------------------------------------------------------
  // Methods
  //----------------------------------------------------------------

  REMOVE_THIS_EXAMPLE_METHOD:

  /** Adds two numbers.
   *
   * @param x The first number.
   * @param y The second number.
   *
   * @return The sum of the two numbers.
   */
  add(x: number, y: number) : number
  {
    this.count += 1;

    return x + y;
  }

} // END class ${CLASS_NAME}
"""

  #----------------------------------------------------------------
  #----------------------------------------------------------------
  cssTemplate = """\
/* *****************************************************************
 * ${CLASS_NAME}.css
 * 2021 Applied Invention, LLC
 *
 * Note that style rules in this file will apply only to the
 * ${CLASS_NAME} component,
 * not to any parents or children.
 * *****************************************************************/

:host
{
  display: block;
}
"""

  #----------------------------------------------------------------
  #----------------------------------------------------------------
  htmlTemplate = """\
<!-- ----------------------------------------------------------------
  -- ${CLASS_NAME}.html
  -- 2021 Applied Invention, LLC
  --
  -- The HTML for the ${CLASS_NAME} component.
  -->

Add ${CLASS_NAME} HTML here
"""
