#-------------------------------------------------------------------
#  NewClass.py
#
#  The NewClass module.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''Allow developers to generate a new class file from a template.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
import inspect
import shlex
import sys
from types import ModuleType
from typing import Dict
from typing import List
from typing import Optional
from .NewClassCommand import NewClassCommand
#
# Import statements go above this line.
#-------------------------------------------------------------------


#-------------------------------------------------------------------
usage = """\
newClass file_type [options...]
newClass help file_type

Creates a starter file for one of the following file types:

FILE_TYPES

Running "newClass help file_type" gives more information about that
file_type's arguments.
"""

#-------------------------------------------------------------------
def execute(templateModule: Optional[ModuleType] = None) -> None:
  '''Execute this command.

  @param templateModule A module that contains all the NewClassCommand classes.
  '''

  if not templateModule:

    # pylint: disable=import-outside-toplevel
    from . import templates
    templateModule = templates

  commandDict: Dict[str, NewClassCommand] = {}

  for commandClassName in templateModule.__dict__:
    commandClass = templateModule.__dict__[commandClassName]

    if (inspect.isclass(commandClass) and
        issubclass(commandClass, NewClassCommand)):
      command = commandClass()
      commandDict[command.name()] = command

  args: List[str] = sys.argv[1:]

  if args and args[0] == '-complete':
    printTabComplete(args[1], commandDict)

  elif len(args) >= 2 and args[0] == 'help' and args[1] in commandDict.keys():
    print(commandDict[args[1]].usageDescription())

  elif not args or args[0] in ('help', '-h', '--help'):
    printHelp(commandDict)

  elif args[0] in commandDict.keys():
    commandDict[args[0]].execute(args[1:])

  else:
    print("Error: unknown command '%s'." % args[0])
    print("Type 'newClass' help' for a list of commands.")
    sys.exit(1)

#-------------------------------------------------------------------
def printTabComplete(completionLine: str,
                     commandDict: Dict[str, NewClassCommand]) -> None:
  '''Print all commands as tab complete suggestions.
  '''

  # The user has asked for auto complete suggestions, so look at
  # the args typed so far.
  args = shlex.split(completionLine)[1:]

  # If the user has typed a blank space and hit tab, he's asking
  # about the next argument.  Add that extra implied 'blank'
  # argument.
  if completionLine and completionLine[-1].isspace():
    args.append('')

  # If the user has already typed a valid command, don't offer any more help.
  # In the future, we may want to expand this to allow each command to
  # offer suggestions.
  if len(args) > 1:
    print()

  else:
    # Suggest all the known commands.
    print(" ".join(sorted(commandDict.keys())))

#-------------------------------------------------------------------
def printHelp(commandDict: Dict[str, NewClassCommand]) -> None:

  names = sorted(commandDict.keys())
  descs = [commandDict[name].shortDescription() for name in names]

  # The length needed to fit all the names, and descriptions.
  maxName = max([len(name) for name in names])
  maxDesc = max([len(desc) for desc in descs])

  formatStr = "  %-" + str(maxName) + "s %-" + str(maxDesc) + "s"

  lines = []

  for name, desc in zip(names, descs):
    lines.append(formatStr % (name, desc))

  helpText = usage.replace("FILE_TYPES", "\n".join(lines))

  print(helpText)
