#-------------------------------------------------------------------
#  VersionFile.py
#
#  The VersionFile class.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the VersionFile class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from .BuildSetup import BuildSetup
from ai.axe.util import StringUtil
import os
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class VersionFile:
  '''File that contains the app version name, generated during the build.
  '''

  # The version used when no version has been specified by the user.
  defaultVersion = 'Development_Build'

  #-----------------------------------------------------------------
  def __init__(self, versionName: str) -> None:
    '''Creates a new VersionFile.
    '''

    # The version name supplied by the user.
    self.userVersionName: str = versionName

  #-----------------------------------------------------------------
  def versionName(self) -> str:
    '''The version that will be written to the file.
    '''
    defaultVersion = VersionFile.defaultVersion

    return self.userVersionName if self.userVersionName else defaultVersion

  #-----------------------------------------------------------------
  def fileName(self) -> str:
    '''The file name (including path).
    '''

    buildSetup = BuildSetup.get()

    rootPackage = buildSetup.rootPackageName()
    rootPackageDir = 'src/python/' + rootPackage.replace('.', '/')

    return '%s/%sVersion.py' % (rootPackageDir, self.appNameCap())

  #-----------------------------------------------------------------
  def appNameCap(self) -> str:

    buildSetup = BuildSetup.get()

    appName = buildSetup.appName()
    appNameCap = StringUtil.capitalize(appName)

    return appNameCap

  #-----------------------------------------------------------------
  def fileText(self) -> str:
    '''The contents of the version file.
    '''

    text = '''# This file is generated.  Do not edit.

# The version of this software.
version = "''' + self.versionName() + '''"
'''

    return text

  #-----------------------------------------------------------------
  def statusMessage(self) -> str:

    msg = 'Generating %sVersion.py with version "%s"...'
    msg = msg % (self.appNameCap(), self.versionName())

    return msg

  #-----------------------------------------------------------------
  def writeFileRunServer(self) -> None:
    '''Writes the file from the runServer script.
    '''

    # If there's an existing version file, leave it alone.
    # If there's no existing file, create one with the default version.

    if not os.path.isfile(self.fileName()):
      self.writeFile()

  #-----------------------------------------------------------------
  def writeFile(self) -> None:
    '''Writes the version file.
    '''

    open(self.fileName(), 'w', encoding='utf-8').write(self.fileText())

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs = ['userVersionName']

    return StringUtil.formatRepr(self, attrs)
