#-------------------------------------------------------------------
#  BuildSetup.py
#
#  The BuildSetup class.
#
#  Copyright 2017 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the BuildSetup class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from typing import Dict
from typing import List
from typing import Optional
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class BuildSetup:
  '''The setup for the Axe build system.

  To use the Axe build functionality, you must create a subclass
  of BuildSetup and register it using BuildSetup.register().

  This must be done before using any other Axe build functionality.
  '''

  #-----------------------------------------------------------------
  # The BuildSetup currently being used.
  setup: Optional['BuildSetup'] = None

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new BuildSetup.
    '''

    pass

  #-----------------------------------------------------------------
  @staticmethod
  def register(buildSetup: 'BuildSetup') -> None:
    '''Registers a buildSetup for the current Axe build.

    @param buildSetup An object that is a BuildSetup subclass.
    '''

    BuildSetup.setup = buildSetup

  #-----------------------------------------------------------------
  def appName(self) -> str:
    '''Returns the name of the app.

    This name is a camel-case string with no spaces.

    This name will be converted from camelCase to ALL_CAPS_AND_UNDERBARS
    to make environment variable names (such as "CROP_DOCTOR_HOME").

    @return A string that is the name of the app, such as "ram" or "cropDoctor".
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def rootPackageName(self) -> str:
    '''Returns the name of the root Python package.

    This name should be a string like "a.b.c.mypackage".

    The default package name is simply the app name.  If you have
    a different pacakge name, you should override this method to
    return that name.

    @return A string that is the name of the root package such as 'ai.axe' or
            'ram'.
    '''

    return self.appName()

  #-----------------------------------------------------------------
  def localDevServerPort(self) -> int:
    '''The port which the local dev server should run on.

    The developer (and browser-based tests) will connect the browser
    to the local dev server using the URL http://localhost:PORT.

    If this app does not have a web server, this value will be ignored.

    @return A port number, such as 8084.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def coverageRelaxations(self) -> Dict[str, int]:
    '''Returns a dict of modules whose coverage requirement has been relaxed.

    @return A {str: int} dictionary of {moduleName: percent_required} of
            % coverage requirements for each module.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  def coverageExemptions(self) -> List[str]:
    '''Returns a dict of modules allowed to have no unit test at all.

    @return A list of string names of modules.
    '''

    raise NotImplementedError()

  #-----------------------------------------------------------------
  @staticmethod
  def get() -> 'BuildSetup':
    '''Returns the buildSetup for the current Axe app.

    @return An object that is a BuildSetup subclass.
    '''

    if BuildSetup.setup:
      return BuildSetup.setup
    else:
      msg = "Error.  The BuildSetup.register() method has not yet been called "
      msg += "to register the BuildSetup class for your Axe app."
      raise ValueError(msg)
