#-------------------------------------------------------------------
#  BuildIdFile.py
#
#  The BuildIdFile class.
#
#  Copyright 2019 Applied Invention, LLC
#-------------------------------------------------------------------

'''The module containing the BuildIdFile class.
'''

#-------------------------------------------------------------------
# Import statements go here.
#
from ai.axe.build.BuildSetup import BuildSetup
from ai.axe.util import StringUtil
from typing import List
import subprocess
#
# Import statements go above this line.
#-------------------------------------------------------------------


#===================================================================
class BuildIdFile:
  '''File that contains a string that identifies a build.
  '''

  # The Build ID used if no build ID can be found.
  defaultBuildId = '(None)'

  #-----------------------------------------------------------------
  def __init__(self) -> None:
    '''Creates a new BuildIdFile.
    '''

    self.cachedBuildId = BuildIdFile.defaultBuildId

  #-----------------------------------------------------------------
  def buildId(self) -> str:
    '''The build ID string that will be written to the file.
    '''
    defaultBuildId = BuildIdFile.defaultBuildId

    # If we've already run 'git describe' successfully, don't run it again.
    if self.cachedBuildId != defaultBuildId:
      return self.cachedBuildId

    cmd = ['git', 'describe', '--tags']
    completed = subprocess.run(cmd, stdout=subprocess.PIPE, encoding='utf-8')

    if completed.returncode == 0:
      self.cachedBuildId = completed.stdout.strip()

    return self.cachedBuildId

  #-----------------------------------------------------------------
  def fileName(self) -> str:
    '''The file name (including path).
    '''

    buildSetup = BuildSetup.get()

    rootPackage = buildSetup.rootPackageName()
    rootPackageDir = 'src/python/' + rootPackage.replace('.', '/')

    return '%s/%sBuildId.py' % (rootPackageDir, self.appNameCap())

  #-----------------------------------------------------------------
  def appNameCap(self) -> str:

    buildSetup = BuildSetup.get()

    appName = buildSetup.appName()
    appNameCap = StringUtil.capitalize(appName)

    return appNameCap

  #-----------------------------------------------------------------
  def fileText(self) -> str:
    '''The contents of the version file.
    '''

    text = '''# This file is generated.  Do not edit.

# The ID of this build.
buildId = "''' + self.buildId() + '''"
'''

    return text

  #-----------------------------------------------------------------
  def statusMessage(self) -> str:

    msg = 'Generating %sBuildId.py with build ID "%s"...'
    msg = msg % (self.appNameCap(), self.buildId())

    return msg

  #-----------------------------------------------------------------
  def writeFile(self) -> None:
    '''Writes the version file.
    '''

    open(self.fileName(), 'w', encoding='utf-8').write(self.fileText())

  #----------------------------------------------------------------
  def __repr__(self) -> str:
    '''Returns a string representation of this object
    '''
    attrs: List[str] = []

    return StringUtil.formatRepr(self, attrs)
