//------------------------------------------------------------------
//  math.ts
//  Copyright 2014 Applied Invention, LLC.
//------------------------------------------------------------------

import { Point } from './Point';
import { XyPoint } from './Point';
import { Angle } from './Angle';

declare var esri: any;

//------------------------------------------------------------------
// Constants
//------------------------------------------------------------------

/** The largest integer value allowed in Javascript.
 */
export const MAX_SAFE_INTEGER: number = Math.pow(2, 53) - 1;

/** The least integer value allowed in Javascript.
 */
export const MIN_SAFE_INTEGER: number = -MAX_SAFE_INTEGER;

/** The number of radians in a degree.
 */
export const radiansPerDegree: number = Math.PI / 180;

/** The number of degrees in a radian.
 */
export const degreesPerRadian: number = 180 / Math.PI;

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Convert degrees to radians.
 *
 * @param degrees A number of degrees.
 *
 * @return A number of radians.
 */
export function degreesToRadians(degrees: number) : number
{
  return degrees * radiansPerDegree;
}

/** Convert radians to degrees.
 *
 * @param radians A number of radians.
 *
 * @return A number of degrees.
 */
export function radiansToDegrees(radians: number) : number
{
  return radians * degreesPerRadian;
}

/** Converts a 'math' angle to a 'map' angle.
 *
 * A 'math' angle starts at 3 o'clock and increases counterclockwise.
 * A 'map' angle starts at 12 o'clock and increases clockwise.
 *
 * @param mathAngle A 'math' angle in radians to convert.
 *
 * @return The equivalent 'map' angle in radians.
 */
export function mathToMapAngle(mathAngle: number) : number
{
  return -mathAngle + (Math.PI / 2);
}

/** Converts a 'map' angle to a 'math' angle.
 *
 * A 'math' angle starts at 3 o'clock and increases counterclockwise.
 * A 'map' angle starts at 12 o'clock and increases clockwise.
 *
 * @param mapAngle A 'map' angle in radians to convert.
 *
 * @return The equivalent 'math' angle in radians.
 */
export function mapToMathAngle(mapAngle: number) : number
{
  return -mapAngle + (Math.PI / 2);
}

/** Scale a point relative to another.
 *
 * A Point is an object that has an 'x' and 'y' property.
 *
 * @param point The Point to be scaled.
 * @param center The center Point that the other point will be scaled
 *                relative to.
 * @param xScale The amount to move the point.  One means no change.
 * @param yScale The amount to move the point.  One means no change.
 *
 * @return A newly created Point object.
 */
export function scalePoint(point: XyPoint,
                           center: XyPoint,
                           xScale: number,
                           yScale: number) : XyPoint
{
  let x = center.x + ((point.x - center.x) * xScale);
  let y = center.y + ((point.y - center.y) * yScale);

  return {'x': x, 'y': y};
}

/** Converts a point from 4326 to web mercator.
 *
 * @param A Point object with an 'x' and 'y' properties.
 *        The x and y must be lon and lat in 4326.
 *
 * @return A newly constructed Point object where x and y are
 *         Web Mercator lon/lat.
 */
export function pointToWebMercator(point: XyPoint) : Point
{
  // Create an ESRI point object.

  let spatialReference = new esri.SpatialReference({'wkid': 4326});
  let point1 = new esri.geometry.Point(point.x,
                                       point.y,
                                       spatialReference);

  point1 = esri.geometry.webMercatorUtils.geographicToWebMercator(point1);

  return point1;
}

/** Round a floating point number.
 *
 * Example:
 *
 *   round(1.005, 2) returns 1.01
 *   round(1.1, 2) returns 1.1
 *
 * @param value The number to round.
 * @param decimals The number of digits of precision required.
 *
 * @return A number.
 */
export function round(value: number, decimals: number) : number
{
  // Appending this to a bar number increases it by 10^decimals.
  let bigSuffix: string = 'e' + decimals;

  // Appending this to a bar number decreases it by 10^decimals.
  let smallSuffix: string = 'e-' + decimals;

  // Make a string version of the big number.
  let bigStr: string = value + bigSuffix;

  // Convert the big number string to a big number;
  let bigNumber: number = +bigStr;

  // Round the big number.
  bigNumber = Math.round(bigNumber);

  // Make a string version of the small number.
  let smallStr: string = bigNumber + smallSuffix;

  // Convert the string to a Number.
  let smallNumber: number = +smallStr;

  return smallNumber;
}

/** Adds two numbers.
 *
 * Can be used in functional programming.
 */
export function add(value1: number, value2: number) : number
{
  return value1 + value2;
}

/** Finds a point perpendicular to the line, and close to the point.
 *
 * @param line1 A Point at the first end of the line.
 * @param line2 A Point at the second end of the line.
 *              The returned point will form a perpendicular line when
 *              connected to this one.
 * @param point A Point that we will find a perpendicular line somewhat near.
 * @param clockwise Boolean Whether the returned point's vector with respect
 *                  to line2 is 90 degrees clockwise or counterclockwise
 *                  from the line1 - line2 vector.
 *
 * @return A Point that forms a perpendicular with line2.
 */
export function perpendicularPoint(line1: Point,
                                   line2: Point,
                                   point: Point,
                                   clockwise: boolean) : Point
{
  let deltaY = line2.y - line1.y;
  let deltaX = line2.x - line1.x;

  let lineAngle = Angle.atan2(deltaY, deltaX);

  let offsetAngle = Angle.degrees(90);
  if (clockwise)
  {
    offsetAngle = offsetAngle.times(-1);
  }
  let pointAngle = lineAngle.plus(offsetAngle);
  let pointSlope = pointAngle.tan();

  // Distance from line2 to point.
  let distance =
    Math.sqrt(Math.pow(line2.y - point.y, 2) + Math.pow(line2.x - point.x, 2));

  // Use Pythagoras to derive x.
  //
  // d - The distance of the point.
  // x - The x coordinate of the point.
  // m - The slope of the point.
  //
  // d^2 = x^2 + (mx)^2
  //
  // d = sqrt( x^2 + (mx)^2 )
  // d = sqrt( x^2 + m^2 x^2 )
  // d = sqrt( x^2 * ( 1 + m@2 ) )
  // d = x * sqrt( 1 + m^2 )
  //
  // x = d / sqrt( 1 + m^2 )

  let x = distance / Math.sqrt(1 + Math.pow(pointSlope, 2));
  let y = pointSlope * x;

  // Because slope is (0, 180), there are 2 possible points,
  // the positive x-derived point and the negative-x derived point.
  //
  // Choose the one whose angle is closest to the point angle that
  // we started with.

  let posPoint = new Point(x, y);
  let negPoint = new Point(-x, -y);

  let posAngle = Angle.atan2(posPoint.y, posPoint.x);
  let negAngle = Angle.atan2(negPoint.y, negPoint.x);

  let posDiff = pointAngle.minus(posAngle);
  let negDiff = pointAngle.minus(negAngle);

  let newPoint = negPoint;
  if (posDiff.radians < negDiff.radians)
  {
    newPoint = posPoint;
  }

  newPoint.x += line2.x;
  newPoint.y += line2.y;

  return newPoint;
}

/** Expands a range of values.
 *
 * @param A list of values.
 *
 * @return A list of values equal in length to the passed-in list.
 */
export function expandRange(values: Array<number>) : Array<number>
{
  if (values.length < 1)
  {
    return values;
  }

  let sum = values.reduce(add, 0);
  let average = sum / values.length;

  // Expand a single value 'away' from the average.
  let expand = function(value: number) : number
  {
    return value + (value - average);
  };

  let newValues = values.map(expand);
  return newValues;
}

/** Expands a range of values.
 *
 * @param A list of values.
 *
 * @return A list of values equal in length to the passed-in list.
 */
export function expandRangeDates(values: Array<Date>) : Array<Date>
{
  // Convert from Dates to numbers.
  let nums: Array<number> = values.map((item: Date) => item.valueOf());

  nums = expandRange(nums);

  // Convert from numbers to Dates.
  values = nums.map((item: number) => new Date(item));

  return values;
}
