//------------------------------------------------------------------
//  http.ts
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
import * as axeNg from './ng';
import { HTTP_INTERCEPTORS } from '@angular/common/http';
import { HttpClient } from '@angular/common/http';
import { HttpErrorResponse } from '@angular/common/http';
import { HttpEvent } from '@angular/common/http';
import { HttpHandler } from '@angular/common/http';
import { HttpHeaders } from '@angular/common/http';
import { HttpInterceptor } from '@angular/common/http';
import { HttpRequest } from '@angular/common/http';
import { HttpResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { ObjectMap } from './types';
import { Observable } from 'rxjs';
import { RequestParams } from './RequestParams';
import { tap } from 'rxjs/operators';
//------------------------------------------------------------------

//------------------------------------------------------------------
// Functions
//------------------------------------------------------------------

/** Creates an HTTP request object.
 *
 * The returned request object will have a method called 'abort()' that
 * allows you to abort an in-progress request.
 *
 * @param url The URL string to request.
 * @param params A RequestParams object containing the request parameters.
 * @param method Optional.  The HTTP method.  If not present, POST will be used.
 * @param formData Optional.  If provided, the specified FormData object
 *                            will be sent as data.  Any parameters
 *                            in the 'params' argument will be sent as well.
 *                            If this parameter is provided, all data will
 *                            be sent as a multipart POST.
 *
 * @return A q promise to do the request.
 */
export function createRequest(url: string,
                              params: RequestParams,
                              method?: string,
                              formData?: FormData) : Observable<any>
{
  if (!method)
  {
    method = 'POST';
  }

  if (!params)
  {
    params = new RequestParams();
  }

  let options: ObjectMap<any> = {};

  if (formData)
  {
    params.addToFormData(formData);

    method = 'POST';

    // In Angular 1, Content-Type=undefined told Angular not to add
    // a Content-Type=application/json header.
    // Not sure how to do this in Angular 2.
    // Discussion here:
    // https://stackoverflow.com/questions/41448070/
    let headers = new HttpHeaders({
      // 'Content-Type': undefined,
    });

    options['body'] = formData;
    options['headers'] = headers;
  }
  else if (method == 'POST')
  {
    let data = params.formatFormUrlEncoded();
    let headers = new HttpHeaders({
      'Content-Type': 'application/x-www-form-urlencoded;charset=UTF-8',
    });

    options['body'] = data;
    options['headers'] = headers;
  }
  else if (method == 'GET' && !params.isEmpty())
  {
    url = url + '?' + params.formatFormUrlEncoded();
  }

  let observable = axeNg.httpClient.request<any>(method, url, options);

  return observable;
}

/** Appends all the files in a FileList to a FormData.
 *
 * @param formData The FormData that the files will be appended to.
 * @param paramName The files will be added with this name.
 * @param fileList The files to be appended to the FormDats.
 */
export function appendFileList(formData: FormData,
                               paramName: string,
                               fileList: FileList) : void
{
  for (let i = 0; i < fileList.length; ++i)
  {
    formData.append(paramName, fileList[i]);
  }
}


/** Handle errors from HTTP request.
 */
@Injectable()
export class AxeNotLoggedInInterceptor implements HttpInterceptor
{
  intercept(request: HttpRequest<any>,
            next: HttpHandler) : Observable<HttpEvent<any>>
  {
    let observable: Observable<HttpEvent<any>> = next.handle(request);

    observable = observable.pipe(tap(handleSuccess, handleError));

    return observable;
  }
}

function handleSuccess(event: HttpEvent<any>) : void
{
  if (event instanceof HttpResponse)
  {
    let response: HttpResponse<any> = event as HttpResponse<any>;

    // If we called a web action and got back the login.html page,
    // then we were redirected because we're not logged in,
    // so send the user to login.html.

    if (axeString.startsWith(response.url, 'app'))
    {
      let contentType: string = response.headers.get('content-type');
      if (axeString.startsWith(contentType, 'text/html'))
      {
        // Redirect to the login page.
        window.location.href = 'login.html';
      }
    }
  }
}

function handleError(value: HttpEvent<any>) : void
{
  if (event instanceof HttpErrorResponse)
  {
    let response: HttpErrorResponse = event as HttpErrorResponse;
    if (response.status == 401)
    {
      // Redirect to the logged-out page.
      window.location.href = 'loggedOut.html';
    }

    writeConsoleError('HTTP error', response);
  }
}

/** Add this as the 'providers' item in the app module set up.
 */
export const httpInterceptorProvider = {
    "provide": HTTP_INTERCEPTORS,
    "useClass": AxeNotLoggedInInterceptor,
    "multi": true,
};

/** A default error handler function.
 *
 * We add this to any HTTP request where the user has not added
 * her own error handler.
 */
export function defaultOnError(response: any) : void
{
  let msg = "A response failed, but you don't have an error handler " +
    "registered for this web action function.";

  writeConsoleError(msg, response);
}

/** Writes information about a failed request to the console.
 */
function writeConsoleError(message: string, response: any) : void
{
  // A status of -1 means the request was aborted.
  let wasAborted = (response.status == -1);

  if (!wasAborted)
  {
    console.error(message, response.status, 'response = ', response);
  }
}

/** Returns the cookies for the current document.
 */
export function documentCookies() : ObjectMap<string>
{
  let cookies: ObjectMap<string> = {};

  let allCookiesStr = document.cookie;

  let cookieStrs = allCookiesStr.split(';');

  for (let i = 0; i < cookieStrs.length; ++i)
  {
    let cookieStr = cookieStrs[i];

    // Remove leading space.
    while (cookieStr.charAt(0) == ' ')
    {
      cookieStr = cookieStr.substr(1);
    }

    let eqIndex = cookieStr.indexOf('=');
    if (eqIndex < 0)
    {
      cookies[cookieStr] = '';
    }
    else
    {
      let name = cookieStr.substr(0, eqIndex);
      let value = '';
      if (cookieStr.length > (eqIndex + 1))
      {
        value = cookieStr.substr(eqIndex + 1);
      }
      cookies[name] = value;
    }
  }
  return cookies;
}
