//------------------------------------------------------------------
//  RequestParams.ts
//  Copyright 2013 AppliedMinds, Inc.
//------------------------------------------------------------------

//------------------------------------------------------------------
import { ClassJsonEncoder } from '../classJson/ClassJsonEncoder';
import { ClassJsonRegistry } from '../classJson/ClassJsonRegistry';
import { Day } from '../date/Day';
import { Duration } from '../date/Duration';
import * as axeDate from "./date";
//------------------------------------------------------------------

/** Parameters for an HTTP request.
 */
export class RequestParams
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The list of parameter keys to send with the request.
   */
  private keys: Array<string> = [];

  /** The list of parameter values to send with the request.
   */
  private values: Array<any> = [];

  /** The encoded list of parameter values to send with the request.
   */
  private encodedValues: Array<string> = [];

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  constructor()
  {
  }

  //------------------------------------------------------------------
  // Static Properties
  //------------------------------------------------------------------

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Adds a parameter to this list.
   *
   * @param key String.  The parameter key.
   * @param value The parameter value.  May be a bool, int, float,
   *              Date, Day, Duration, string,
   *              object, or list of any of the above.
   */
  add(key: string, value: any) : void
  {
    if (typeof value == 'object' && value instanceof Array)
    {
      this.addList(key, <Array<any>>value);
    }
    else
    {
      this.keys.push(key);
      this.pushValue(value);
    }
  }

  /** Adds a list type parameter to this list.
   *
   * @param key String.  The parameter key.
   * @param value The parameter value.  Must be a list.
   */
  addList(key: string, valueList: Array<any>) : void
  {
    for (let i = 0; i < valueList.length; ++i)
    {
      let value = valueList[i];

      this.keys.push(key);
      this.pushValue(value);
    }
  }

  /** Adds values from the map, if they appear in the validKeys array.
   */
  addFromMap(map: any, validKeys: Array<string>) : void
  {
    for (let i = 0; i < validKeys.length; ++i)
    {
      let key = validKeys[i];
      this.add(key, map[key]);
    }
  }

  /** Adds the key/values from this to the specified FormData.
   */
  addToFormData(formData: FormData) : void
  {
    for (let i = 0; i < this.keys.length; ++i)
    {
      formData.append(this.keys[i], this.encodedValues[i]);
    }
  }

  /** Returns true if this list has no parameters at all.
   */
  isEmpty() : boolean
  {
    return this.keys.length == 0;
  }

  /** Returns a parameter value.
   *
   * @param key String.  The key of the parameter to return.
   *
   * @return The parameter value.
   */
  get(key: string) : any
  {
    for (let i = 0; i < this.keys.length; ++i)
    {
      if (this.keys[i] == key)
      {
        return this.values[i];
      }
    }

    throw Error("Invalid RequestParams key: " + key);
  }

  /** Encodes a parameter value.
   *
   * @param value The value to encode.
   */
  encode(value: any) : string
  {
    if (value === null)
    {
      return '';
    }
    else if (typeof value == 'object' && value instanceof Day)
    {
      return axeDate.formatDate(value);
    }
    else if (typeof value == 'object' && value instanceof Date)
    {
      return axeDate.formatDateTime(value);
    }
    else if (typeof value == 'object' && value instanceof Duration)
    {
      return this.encode(Math.trunc(value.milliseconds));
    }
    else if (typeof value == 'object' &&
             ClassJsonRegistry.registry.isRegisteredClass(value.constructor))
    {
      return (new ClassJsonEncoder()).encode(value);
    }
    else
    {
      return '' + value;
    }
  }

  /** Pushes a value in both raw and encoded forms.
   */
  pushValue(value: any) : void
  {
    this.values.push(value);
    this.encodedValues.push(this.encode(value));
  }

  /** Returns these params in application/x-www-form-urlencoded content type.
   */
  formatFormUrlEncoded() : string
  {
    let ret = '';

    for (let i = 0; i < this.keys.length; ++i)
    {
      if (i > 0)
      {
        ret += '&';
      }
      ret += encodeURIComponent(this.keys[i]);
      ret += '=';
      ret += encodeURIComponent(this.encodedValues[i]);
    }

    return ret;
  }

  /** Return a RequestParams object for the params encoded in the string.
   */
  static parseFormUrlEncoded(paramsString: string) : RequestParams
  {
    let ret = new RequestParams();

    let pairList = paramsString.split('&');
    for (let i = 0; i < pairList.length; ++i)
    {
      let keyValue = pairList[i].split('=');
      let key = decodeURIComponent(keyValue[0]);
      let value = decodeURIComponent(keyValue[1]);

      ret.add(key, value);
    }

    return ret;
  }

} // END class RequestParams

