//------------------------------------------------------------------
//  Duration.ts
//  Copyright 2014 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
//------------------------------------------------------------------

/** A duration in time.
 *
 * This class models a time duration as an integer number of
 * milliseconds.
 */
export class Duration
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** Integer number of milliseconds;
   */
  milliseconds: number;

  //----------------------------------------------------------------
  // Initialization
  //----------------------------------------------------------------

  /** Initializes a new Duration object.
   */
  constructor(milliseconds: number)
  {
    this.milliseconds = milliseconds;

    if (milliseconds != Math.trunc(milliseconds))
    {
      let msg = "Duration requires an integer number of milliseconds.  ";
      msg += "Received: " + milliseconds;
      throw new Error(msg);
    }
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Adds this duration to a Date.
   *
   * @param date The date to add this to.
   *
   * @return A new Date that is the sum of the input date and this duration.
   */
  addTo(date: Date) : Date
  {
    return new Date(date.getTime() + this.milliseconds);
  }

  /** Subtracts this duration from a Date.
   *
   * @param date The date to add this to.
   *
   * @return A new Date, the difference of the input date and this duration.
   */
  subtractFrom(date: Date) : Date
  {
    return new Date(date.getTime() - this.milliseconds);
  }

  /** Returns a new Duration is the negative of this one.
   *
   * @return A new Duration that is the negative of this one.
   */
  negative() : Duration
  {
    return new Duration(-this.milliseconds);
  }

  /** Returns a new Duration that is the product of this and a number.
   */
  times(multiple: number) : Duration
  {
    let value = this.milliseconds * multiple;
    value = Math.round(value);

    return new Duration(value);
  }

  /** Returns a new Duration that is the result of dividing this by a number.
   */
  dividedBy(divisor: number) : Duration
  {
    let value = this.milliseconds / divisor;
    value = Math.round(value);

    return new Duration(value);
  }

  /** Gets the number of whole days contained in this duration.
   */
  getDays() : number
  {
    let ret = this.milliseconds / 1000 / 60 / 60 / 24;
    ret = Math.floor(ret);
    return ret;
  }

  /** Gets the number of whole hours contained in this duration.
   */
  getHours() : number
  {
    let ret = this.milliseconds / 1000 / 60 / 60;
    ret = Math.floor(ret);
    ret = ret % 24;
    return ret;
  }

  /** Gets the number of whole minutes contained in this duration.
   */
  getMinutes() : number
  {
    let ret = this.milliseconds / 1000 / 60;
    ret = Math.floor(ret);
    ret = ret % 60;
    return ret;
  }

  /** Gets the number of whole seconds contained in this duration.
   */
  getSeconds() : number
  {
    let ret = this.milliseconds / 1000;
    ret = Math.floor(ret);
    ret = ret % 60;
    return ret;
  }

  /** Gets the number of milliseconds contained in this duration.
   */
  getMilliseconds() : number
  {
    let ret = this.milliseconds;
    ret = ret % 1000;
    return ret;
  }

  /** Creates a new duration that is the difference of two dates.
   *
   * @param date1 The date to subtract from.
   * @param date2 The date to subtract.
   *
   * @return A duration that is date1 - date2.
   */
  static difference(date1: Date, date2: Date) : Duration
  {
    return new Duration(date1.getTime() - date2.getTime());
  }

  /** Returns the number of hours that this duration is equivalent to.
   *
   * @return A floating-point number of hours.
   */
  toHours() : number
  {
    let millisecondsPerHour = 60 * 60 * 1000;
    return this.milliseconds / millisecondsPerHour;
  }

  /** Returns a duration eqivalent to the specified number of hours.
   *
   * @param hours A floating-point number of hours.  If contains
   *              a fractional number of milliseconds, will be rounded
   *              to the nearest millisecond.
   *
   * @return A Duration object.
   */
  static fromHours(hours: number) : Duration
  {
    // Convert to integer milliseconds.
    let millisecondsPerHour = 60 * 60 * 1000;
    let milliseconds = hours *  millisecondsPerHour;
    milliseconds = Math.round(milliseconds);

    return new Duration(milliseconds);
  }

  /** Returns a duration eqivalent to the specified number of days.
   *
   * @param days A floating-point number of days.  If contains
   *             a fractional number of milliseconds, will be rounded
   *             to the nearest millisecond.
   *
   * @return A Duration object.
   */
  static fromDays(days: number) : Duration
  {
    // Convert to integer milliseconds.
    let millisecondsPerDay = 24 * 60 * 60 * 1000;
    let milliseconds = days *  millisecondsPerDay;
    milliseconds = Math.round(milliseconds);

    return new Duration(milliseconds);
  }

  /** Creates the JSON-ready form of this object.
   */
  static toJson(obj: Duration) : string
  {
    return axeString.format(obj.milliseconds) + "ms";
  }

  /** Returns a new Duration object from a JSON-ified duration.
   */
  static fromJson(jsonObj: string) : Duration
  {
    if (!axeString.endsWith(jsonObj, "ms"))
    {
      let msg = "Error parsing JSON Duration.  ";
      msg += "String should end in 'ms'.  ";
      msg += "String: " + axeString.format(jsonObj);
      console.error(msg);
      throw new Error(msg);
    }

    jsonObj = axeString.substring(jsonObj, 0, -2);
    return new Duration(parseInt(jsonObj));
  }

  /** Returns a primitive representation of this object.
   *
   * Returns the total number of milliseconds as a number.
   */
  valueOf() : number
  {
    return this.milliseconds;
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let ret = "Duration(days=" + this.getDays() + " hours=" + this.getHours();
    ret += " minutes=" + this.getMinutes() + " seconds=" + this.getSeconds();
    ret += " msec=" + this.getMilliseconds() + " total=" + this.milliseconds;
    ret += ")";

    return ret;
  }

} // END class Duration
