//------------------------------------------------------------------
//  TestJsonMap.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../../../util/string";
import { ClassJsonDesc } from '../../ClassJsonDesc';
import { ClassJsonRegistry } from '../../ClassJsonRegistry';
import { JsonLink } from '../JsonLink';
import { JsonList } from '../JsonList';
import { JsonMap } from '../JsonMap';
import { JsonObj } from '../JsonObj';
import { JsonPrimitiveType } from '../JsonPrimitiveType';
import { Map } from "../../../util/Map";
import { MapKeyType } from "../../../util/Map";
import { ObjectMap } from "../../../util/types";
import { UnitTest } from '../../../unittest/UnitTest';
import { UnitTestRunner } from '../../../unittest/UnitTestRunner';
//------------------------------------------------------------------

/** Unit test for the JsonMap class.
 */
export class TestJsonMap extends UnitTest
{
  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonMap object.
   */
  constructor()
  {
    super();
  }

  //------------------------------------------------------------------
  // Test Methods (name starts with 'test')
  //------------------------------------------------------------------

  /** A wrong dict type should throw an exception.
   */
  testBadDictType() : void
  {
    let jsonType = new JsonMap(new JsonPrimitiveType('int'),
                               new JsonPrimitiveType('bool'));

    let errorMsg = jsonType.validate('bad value');
    this.assertNotNull('bad value', errorMsg);

    errorMsg = jsonType.validateJson('bad value');
    this.assertNotNull('bad value', errorMsg);
  }

  /** A wrong dict item type should throw an exception.
   */
  testBadDictItemType() : void
  {
    let desc: ClassJsonDesc = new ClassJsonDesc("TestClass");
    desc.addField('foo', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass
    {
      foo: number;

      constructor(foo: number)
      {
        this.foo = foo;
      }

      static fromJson(src: ObjectMap<any>) : TestClass
      {
        return new TestClass(src['foo']);
      }

      static toJson(src: TestClass) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass);

    class TestClassMapKeyType implements MapKeyType<TestClass>
    {
      typeName: string;

      constructor()
      {
        this.typeName = "TestClass";
      }

      encode(key: TestClass) : string
      {
        return "" + key.foo;
      }
    }
    JsonMap.setMapKeyType("TestClass", new TestClassMapKeyType());

    desc = new ClassJsonDesc("TestClass2");
    desc.addField('foo', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClass2
    {
      foo: number;

      constructor(foo: number)
      {
        this.foo = foo;
      }

      static fromJson(src: ObjectMap<any>) : TestClass2
      {
        return new TestClass2(src['foo']);
      }

      static toJson(src: TestClass2) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClass2);


    let jsonType = new JsonMap(new JsonObj('TestClass'),
                               new JsonObj('TestClass2'));

    class TestClassKeyType implements MapKeyType<TestClass>
    {
      typeName: string;

      constructor()
      {
        this.typeName = "TestClass";
      }

      encode(key: TestClass) : string
      {
        return "" + key.foo;
      }
    }


    let theDict = new Map<TestClass, TestClass2>(new TestClassKeyType());

    theDict.put(new TestClass(1), new TestClass2(12));
    theDict.put(new TestClass(2), new TestClass2(13));

    let jsonTestClass1 = {'_class': 'TestClass', 'foo': 1};
    let jsonTestClass2 = {'_class': 'TestClass', 'foo': 2};
    let jsonTestClass3 = {'_class': 'TestClass', 'foo': 3};
    let jsonTestClass12 = {'_class': 'TestClass2', 'foo': 12};
    let jsonTestClass13 = {'_class': 'TestClass2', 'foo': 13};
    let jsonTestClass14 = {'_class': 'TestClass2', 'foo': 14};

    let jsonDict: any = [
      [jsonTestClass1, jsonTestClass12],
      [jsonTestClass2, jsonTestClass13],
    ];

    let errorMsg = jsonType.validate(theDict);
    this.assertNull('valid', errorMsg);

    errorMsg = jsonType.validateJson(jsonDict);
    this.assertNull('valid JSON', errorMsg);

    // Test a value of the wrong type.

    theDict.put(new TestClass(3), new TestClass(14));

    errorMsg = jsonType.validate(theDict);
    this.assertNotNull('wrong value type', errorMsg);

    jsonDict.push([jsonTestClass3, jsonTestClass1]);

    errorMsg = jsonType.validateJson(jsonDict);
    this.assertNotNull('wrong value type JSON', errorMsg);

    // Test a key of the wrong type.

    theDict.removeAll();
    theDict.put(new TestClass(1), new TestClass2(12));
    theDict.put(new TestClass(2), new TestClass2(13));
    theDict.put(new TestClass2(3), new TestClass2(14));

    errorMsg = jsonType.validate(theDict);
    this.assertNotNull('wrong key type', errorMsg);

    jsonDict[4] = [jsonTestClass12, jsonTestClass14];

    errorMsg = jsonType.validateJson(jsonDict);
    this.assertNotNull('wrong key type JSON', errorMsg);
  }

  /** A wrong dict item type should throw an exception.
   */
  testBadDictItemTypeJson() : void
  {
    let jsonType = new JsonMap(new JsonPrimitiveType('int'),
                               new JsonPrimitiveType('float'));

    let jsonDict: Array<any> = [[1, 1.1], [2, 2.2]];

    let errorMsg = jsonType.validateJson(jsonDict);
    this.assertNull('valid', errorMsg);

    // Test wrong length item sub-list.
    jsonDict = [[1, 1.1, 'wrong'], [2, 2.2]];

    errorMsg = jsonType.validateJson(jsonDict);
    this.assertNotNull('wrong length item sub-list', errorMsg);

    // Test a value of the wrong type.
    jsonDict = [[1, 1.1], [2, 'bad type']];

    errorMsg = jsonType.validateJson(jsonDict);
    this.assertNotNull('value of the wrong type', errorMsg);

    // Test a key of the wrong type.
    jsonDict = [[1, 1.1], ['bad type', 2.22]];

    errorMsg = jsonType.validateJson(jsonDict);
    this.assertNotNull('key of the wrong type', errorMsg);
  }

  /** Test the encode() method.
   */
  testEncode() : void
  {
    let jsonType = new JsonMap(new JsonPrimitiveType('int'),
                               new JsonPrimitiveType('float'));

    this.assertEqual('validate null', null, jsonType.validate(null));
    this.assertEqual('encode null', null, jsonType.encode(null));

    let jsonDict = [[1, 1.1], [2, 2.2]];

    let objDict = Map.createIntMap<number>();
    objDict.put(1, 1.1);
    objDict.put(2, 2.2);

    let actual = jsonType.encode(objDict);

    this.assertEqual('encode', jsonDict, actual);
  }

  /** Test the decode() method.
   */
  testDecode() : void
  {
    let jsonType = new JsonMap(new JsonPrimitiveType('int'),
                               new JsonPrimitiveType('float'));

    this.assertEqual('validate null', null, jsonType.validateJson(null));
    this.assertEqual('decode null', null, jsonType.decode(null));

    let jsonDict = [[1, 1.1], [2, 2.2]];

    let objDict = Map.createIntMap<number>();
    objDict.put(1, 1.1);
    objDict.put(2, 2.2);

    let actual = jsonType.decode(jsonDict);

    this.assertEqual('decode', objDict, actual);
  }

  /** Test the decodeLinks() method.
   */
  testDecodeLinks() : void
  {
    // A band is a list of musicians and instruments.
    // Every musician has an instrument.

    let desc: ClassJsonDesc = new ClassJsonDesc("TestClassInstrument");
    desc.addField('id', new JsonPrimitiveType('int'));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassInstrument
    {
      id: number;

      constructor(id: number)
      {
        this.id = id;
      }

      static fromJson(src: ObjectMap<any>) : TestClassInstrument
      {
        return new TestClassInstrument(<number>src['id']);
      }

      static toJson(src: TestClassInstrument) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassInstrument);

    desc = new ClassJsonDesc("TestClassMusician");
    desc.addField('id', new JsonPrimitiveType('int'));
    desc.addField('instrument', new JsonLink("TestClassInstrument",
                                             "../instruments",
                                             ["id"]));
    ClassJsonRegistry.registry.addDesc(desc);

    class TestClassMusician
    {
      id: number;
      instrument: TestClassInstrument;
      constructor(id: number, instrument: TestClassInstrument)
      {
        this.id = id;
        this.instrument = instrument;
      }

      static fromJson(src: ObjectMap<any>) : TestClassMusician
      {
        return new TestClassMusician(<number>src['id'],
                                     <TestClassInstrument>src['instrument']);
      }

      static toJson(src: TestClassMusician) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassMusician);

    desc = new ClassJsonDesc("TestClassBand");
    desc.addField('musicians',
                  new JsonMap(new JsonPrimitiveType('string'),
                              new JsonObj("TestClassMusician")));
    desc.addField('instruments',
                  new JsonList(new JsonObj("TestClassInstument")));

    class TestClassBand
    {
      musicians: Map<string, TestClassMusician>;
      instruments: Array<TestClassInstrument>;
      constructor(theMusicians: Map<string, TestClassMusician>,
                  theInstruments: Array<TestClassInstrument>)
      {
        this.musicians = theMusicians;
        this.instruments = theInstruments;
      }

      static fromJson(src: ObjectMap<any>) : TestClassBand
      {
        return new TestClassBand(
            <Map<string, TestClassMusician>>src['musicians'],
            <Array<TestClassInstrument>>src['instruments']
        );
      }

      static toJson(src: TestClassBand) : ObjectMap<any>
      {
        return src;
      }
    }

    ClassJsonRegistry.registry.register(desc.className, TestClassBand);

    let instruments = [
      new TestClassInstrument(20),
      new TestClassInstrument(21),
      new TestClassInstrument(22),
    ];

    let musicians = Map.createStringMap<TestClassMusician>();
    musicians.put('Jane', new TestClassMusician(10, instruments[2]));
    musicians.put('Fred', new TestClassMusician(11, instruments[0]));

    let band = new TestClassBand(musicians, instruments);

    // Link are unresolved, so instead of pointers to objects,
    // they are the object IDs.

    for (let musician of musicians.values())
    {
      (<any>musician.instrument) = [musician.instrument.id];
    }

    let jsonType = new JsonMap(new JsonPrimitiveType('string'),
                               new JsonObj('TestClassMusician'));
    let parents: Array<object> = [band];

    this.assertNull('null', jsonType.decodeLinks(parents, null));

    jsonType.decodeLinks(parents, musicians);

    this.assertEqual('Jane', instruments[2], musicians.get('Jane').instrument);
    this.assertEqual('Frame', instruments[0], musicians.get('Fred').instrument);
  }

  //------------------------------------------------------------------
  // Private Helper Methods
  //------------------------------------------------------------------

} // END class TestJsonMap

//------------------------------------------------------------------
// Register the test.
UnitTestRunner.add(new TestJsonMap());
