//------------------------------------------------------------------
//  JsonObj.ts
//  Copyright 2016 Applied Invention, LLC
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeArray from '../../util/array';
import * as axeClasses from '../../util/classes';
import * as axeString from "../../util/string";
import { ClassJsonDesc } from '../ClassJsonDesc';
import { ClassJsonRegistry } from '../ClassJsonRegistry';
import { JsonType } from './JsonType';
import { JsonTypeError } from './JsonTypeError';
import { Jsonable } from '../classJson';
import { ObjectMap } from '../../util/types';
import { assert } from '../classJson';
//------------------------------------------------------------------

/** Marks a class attribute as an object of a ClassJson type.
 */
export class JsonObj extends JsonType
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The name of the class that this object encode/decode.
   */
  className: string;

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new JsonObj object.
   */
  constructor(className: string)
  {
    super();

    this.className = className;
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Returns the class of the object that this encodes/decodes.
   */
  objClass() : Jsonable<any, any>
  {
    return ClassJsonRegistry.registry.registeredClass(this.className);
  }

  /** Checks that the specified value can be converted to JSON.
   *
   * @param value The value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validate(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    let clazz = value.constructor;
    let desc = ClassJsonRegistry.registry.getDescForClass(clazz);

    if (this.className != desc.className &&
        !axeArray.contains(desc.baseClassNames(), this.className))
    {
      let msg = ('must be of type ' + this.className +
                 ' but the value is of type ' + this.classNameOf(value) +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }

    if (desc.isAbstract)
    {
      let msg = ('cannot be encoded because it is of abstract class ' +
                 desc.className + '.  That class is decorated with a ' +
                 '@ClassJsonClass with isAbstract=True.  An abstract class ' +
                 'cannot be encoded or decoded.');
      return new JsonTypeError(msg);
    }

    let missingProps = this.checkObjProps(desc.getFieldNames(), value);

    if (missingProps.length > 0)
    {
      let msg = 'has missing attributes: ' + missingProps.join(', ');
      return new JsonTypeError(msg);
    }

    for (let propName of desc.getFieldNames())
    {
      let jsonType = desc.getFieldType(propName);
      let propMsg = jsonType.validate(value[propName]);
      if (propMsg)
      {
        propMsg.prependPath(propName);
        return propMsg;
      }
    }

    return null;
  }

  /** Checks that the specified JSON string can be converted to an object.
   *
   * @param value The JSON value to validate.
   *
   * @return None if the value is OK, or a JsonTypeError if there is a problem.
   */
  validateJson(value: any) : JsonTypeError
  {
    if (value === null)
    {
      return null;
    }

    let propDict: ObjectMap<any> = <ObjectMap<any>>value;

    if (typeof propDict != 'object')
    {
      let msg = ('is of type "' + this.className + '", but the JSON object ' +
                 'was type "' + this.classNameOf(propDict) + '".');
      return new JsonTypeError(msg);
    }

    if (!('_class' in propDict))
    {
      return new JsonTypeError("is missing attribute '_class'");
    }

    let className: string = <string>propDict['_class'];

    if (!ClassJsonRegistry.registry.isRegistered(className))
    {
      let msg = "has _class=" + className + ", but that class doesn't have ";
      msg += "a @ClassJsonClass decoration.";
      return new JsonTypeError(msg);
    }

    let desc = ClassJsonRegistry.registry.getDesc(className);

    if (this.className != desc.className &&
        !axeArray.contains(desc.baseClassNames(), this.className))
    {
      let msg = ('must be of type ' + this.className +
                 ' but the value is of type ' + desc.className +
                 '.  Value: ' + axeString.format(value));
      return new JsonTypeError(msg);
    }

    if (desc.isAbstract)
    {
      let msg = ('is an object of abstract class ' +
                 desc.className + '.  That class is decorated with a ' +
                 '@ClassJsonClass with isAbstract=True.  An abstract class ' +
                 'cannot be encoded or decoded.');
      return new JsonTypeError(msg);
    }

    let extraMissing = this.checkJsonObjProps(desc.getFieldNames(), propDict);
    let extra = extraMissing[0];
    let missing = extraMissing[1];

    if (extra.length > 0 || missing.length > 0)
    {
      let msg = 'has invalid properties. ';

      if (extra.length > 0)
      {
        msg += 'The following extra properties were found: ' + extra;
      }
      if (extra.length > 0 && missing.length > 0)
      {
        msg += '\n';
      }
      if (missing.length > 0)
      {
        msg += 'The following properties were missing: ' + missing;
      }
      return new JsonTypeError(msg);
    }

    for (let propName of desc.getFieldNames())
    {
      let jsonType = desc.getFieldType(propName);

      let propMsg = jsonType.validateJson(propDict[propName]);
      if (propMsg)
      {
        propMsg.prependPath(propName);
        return propMsg;
      }
    }

    return null;
  }

  /** Encodes a value into JSON-ready value.
   */
  encode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let obj = value;

    let ret: ObjectMap<any> = {};

    let clazz = obj.constructor;
    let desc = ClassJsonRegistry.registry.getDescForClass(clazz);

    assert(this.className == desc.className ||
           axeArray.contains(desc.baseClassNames(), this.className),
           'is not class or subclass', this.className, desc.className);
    assert(!desc.isAbstract, 'isAbstract');

    ret['_class'] = desc.className;

    for (let propName of desc.getFieldNames())
    {
      let jsonType = desc.getFieldType(propName);

      assert(propName in obj, 'Missing property', propName);

      let propValue = obj[propName];
      ret[propName] = jsonType.encode(propValue);
    }

    return ret;
  }

  /** Decodes a value from a JSON-ready value.
   */
  decode(value: any) : any
  {
    if (value === null)
    {
      return null;
    }

    let propDict: ObjectMap<any> = <ObjectMap<any>>value;

    assert('_class' in propDict);

    let className = <string>propDict['_class'];
    assert(ClassJsonRegistry.registry.isRegistered(className));

    let clazz: Jsonable<any, any> =
      ClassJsonRegistry.registry.registeredClass(className);
    let desc: ClassJsonDesc = ClassJsonRegistry.registry.getDesc(className);

    assert(this.className == desc.className ||
           axeArray.contains(desc.baseClassNames(), this.className),
           'is not class or subclass', this.className, desc.className);
    assert(!desc.isAbstract, 'isAbstract');

    let extraMissing = this.checkJsonObjProps(desc.getFieldNames(), propDict);
    let extra = extraMissing[0];
    let missing = extraMissing[1];
    assert(extra.length == 0, extra);
    assert(missing.length == 0, missing);

    // Decode each field value.
    let decodedDict: ObjectMap<any> = {};
    for (let propName of desc.getFieldNames())
    {
      let propType = desc.getFieldType(propName);
      let propValue = propDict[propName];

      assert(!propType.validateJson(propValue),
             propType.validateJson(propValue));

      decodedDict[propName] = propType.decode(propValue);
    }

    let obj = clazz.fromJson(decodedDict);

    return obj;
  }

  /** Decodes any links in a JSON-ready value.
   */
  decodeLinks(parents: Array<object>, value: object) : object
  {
    if (value === null)
    {
      return null;
    }

    let obj = <ObjectMap<any>>value;

    let clazz = obj.constructor;
    let desc = ClassJsonRegistry.registry.getDescForClass(clazz);

    assert(this.className == desc.className ||
           axeArray.contains(desc.baseClassNames(), this.className),
           'is not class or subclass', this.className, desc.className);
    assert(!desc.isAbstract, 'isAbstract');

    let childParents: Array<object> = parents.concat([value]);

    for (let propName of desc.getFieldNames())
    {
      let propType = desc.getFieldType(propName);
      let oldValue = obj[propName];

      let newValue = propType.decodeLinks(childParents, oldValue);
      if (oldValue !== newValue)
      {
        obj[propName] = newValue;
      }
    }

    return value;
  }

  /** Returns a list of missing properties.
   */
  checkObjProps(propNameList: Array<string>, obj: any) : Array<string>
  {
    let missing: Array<string> = [];

    for (let name of propNameList)
    {
      if (!obj.hasOwnProperty(name))
      {
        missing.push(name);
      }
    }
    return missing;
  }

  /** Returns a [[extra], [missing]] list of lists of properties.
   */
  checkJsonObjProps(propNameList: Array<string>,
                    propDict: ObjectMap<any>) : Array<Array<string>>
  {
    let expectedNames = propNameList;

    let actualNames: Array<string> = [];
    for (let propName in propDict)
    {
      actualNames.push(propName);
    }

    let missing: Array<string> = [];
    for (let name of expectedNames)
    {
      if (!axeArray.contains(actualNames, name))
      {
        missing.push(name);
      }
    }

    let extra: Array<string> = [];
    for (let name of actualNames)
    {
      if (name != '_class' && !axeArray.contains(expectedNames, name))
      {
        extra.push(name);
      }
    }

    return [extra, missing];
  }

  /** Returns the name of the class of an object.
   */
  classNameOf(obj: any) : string
  {
    return axeClasses.className(obj);
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      'typeName'
    ];
    return axeString.formatObject("JsonObj", this, propertyNames);
  }

} // END class JsonObj
