//------------------------------------------------------------------
//  ClassJsonRegistry.ts
//  Copyright 2016 Applied Invention, LLC.
//------------------------------------------------------------------

//------------------------------------------------------------------
import * as axeString from "../util/string";
import { createJsonableMap } from './classJson';
import { Jsonable } from './classJson';
import { ClassJsonDesc } from './ClassJsonDesc';
import { Map } from '../util/Map';
//------------------------------------------------------------------

/** The ClassJson type registry.
 *
 * A user can register a specific Javascript classes that a particular
 * ClassJson type should be decoded into.
 */
export class ClassJsonRegistry
{
  //----------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------

  /** The one and only global registry.
   */
  static registry: ClassJsonRegistry = new ClassJsonRegistry();

  /** The schema to use for encoding each ClassJson class.
   */
  private descs: Map<string, ClassJsonDesc> =
    Map.createStringMap<ClassJsonDesc>();

  /** The javascript class to use for each ClassJson class.
   */
  private jsClasses: Map<string, Jsonable<any, any>> =
    Map.createStringMap<Jsonable<any, any>>();

  /** The ClassJson class name for each javascript class.
   */
  private classNames: Map<Jsonable<any, any>, string> = createJsonableMap();

  //----------------------------------------------------------------
  // Creation
  //----------------------------------------------------------------

  /** Creates a new ClassJsonRegistry object.
   */
  constructor()
  {
  }

  //------------------------------------------------------------------
  // Methods
  //------------------------------------------------------------------

  /** Registers a class to be used for a classJsonClass type.
   *
   * @param classJsonClass The name of the class (stored by ClassJson
   *                       as the '_class' field on each JSON object).
   * @param clazz An object of this class will be created when
   *              the specified classJsonClass is decoded.
   * @param overwrite Whether an existing registrations should be replaced.
   */
  register(classJsonClass: string,
           clazz: Jsonable<any, any>,
           overwrite: boolean = true) : void
  {
    // If we're not overwriting, return if a binding already exists.

    if ((this.jsClasses.hasKey(classJsonClass) ||
         this.classNames.hasKey(clazz)) &&
        !overwrite)
    {
      return;
    }

    // Remove old bindings for either the class or class name,
    // if any exist.

    if (this.classNames.hasKey(clazz))
    {
      this.jsClasses.remove(this.classNames.get(clazz));
    }
    if (this.jsClasses.hasKey(classJsonClass))
    {
      this.classNames.remove(this.jsClasses.get(classJsonClass));
    }

    // Add the new bindings.

    this.jsClasses.put(classJsonClass, clazz);
    this.classNames.put(clazz, classJsonClass);
  }

  /** Returns true if the specified class name has been registered.
   */
  isRegistered(className: string) : boolean
  {
    return this.jsClasses.hasKey(className);
  }

  /** Returns true if the specified class has been registered.
   */
  isRegisteredClass(clazz: Jsonable<any, any>) : boolean
  {
    return this.classNames.hasKey(clazz);
  }

  /** Returns the ClassJson class name for the specified registered class.
   *
   * @param clazz A class that has been registered with the register() method.
   *
   * @return The ClassJson class name that the class was registered for.
   */
  className(clazz: any) : string
  {
    if (!this.classNames.hasKey(clazz))
    {
      let msg: string = "Error: class not registered.  Class: ";
      console.error(msg, clazz);
      throw new Error(msg + clazz);
    }
    return this.classNames.get(clazz);
  }

  /** Returns the class registered for the specified ClassJson class name.
   *
   * @param className A ClassJson class name that has been registered
   *                  with the register() method.
   *
   * @return The class that the class name was registered for.
   */
  registeredClass(className: string) : Jsonable<any, any>
  {
    if (!this.jsClasses.hasKey(className))
    {
      let msg: string = "Error: class name not registered.  Class name: ";
      console.error(msg, className);
      throw new Error(msg + className);
    }
    return this.jsClasses.get(className);
  }

  /** Adds a ClassJson class schema to this registry.
   */
  addDesc(desc: ClassJsonDesc) : void
  {
    this.descs.put(desc.className, desc);
  }

  /** Returns the ClassJson class schema for the specified class name.
   */
  getDesc(className: string) : ClassJsonDesc
  {
    if (!this.descs.hasKey(className))
    {
      let msg: string = "Error: class desc not registered.  Class name: ";
      console.error(msg, className);
      throw new Error(msg + className);
    }
    return this.descs.get(className);
  }

  /** Returns the ClassJson class schema for the specified class.
   */
  getDescForClass(clazz: any) : ClassJsonDesc
  {
    return this.getDesc(this.className(clazz));
  }

  /** Returns a string representation of this object.
   */
  toString() : string
  {
    let propertyNames: Array<string> = [
      'jsClasses',
      'descs',
    ];
    return axeString.formatObject("ClassJsonRegistry", this, propertyNames);
  }

} // END class ClassJsonRegistry
