/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { HarnessEnvironment } from '@angular/cdk/testing';
import * as webdriver from 'selenium-webdriver';
import { SeleniumWebDriverElement } from './selenium-web-driver-element';
/** The default environment options. */
const defaultEnvironmentOptions = {
    queryFn: async (selector, root) => root().findElements(webdriver.By.css(selector)),
};
/**
 * This function is meant to be executed in the browser. It taps into the hooks exposed by Angular
 * and invokes the specified `callback` when the application is stable (no more pending tasks).
 */
function whenStable(callback) {
    Promise.all(window.frameworkStabilizers.map(stabilizer => new Promise(stabilizer))).then(callback);
}
/**
 * This function is meant to be executed in the browser. It checks whether the Angular framework has
 * bootstrapped yet.
 */
function isBootstrapped() {
    return !!window.frameworkStabilizers;
}
/** Waits for angular to be ready after the page load. */
export async function waitForAngularReady(wd) {
    await wd.wait(() => wd.executeScript(isBootstrapped));
    await wd.executeAsyncScript(whenStable);
}
/** A `HarnessEnvironment` implementation for WebDriver. */
export class SeleniumWebDriverHarnessEnvironment extends HarnessEnvironment {
    constructor(rawRootElement, options) {
        super(rawRootElement);
        this._options = { ...defaultEnvironmentOptions, ...options };
        this._stabilizeCallback = () => this.forceStabilize();
    }
    /** Gets the ElementFinder corresponding to the given TestElement. */
    static getNativeElement(el) {
        if (el instanceof SeleniumWebDriverElement) {
            return el.element();
        }
        throw Error('This TestElement was not created by the WebDriverHarnessEnvironment');
    }
    /** Creates a `HarnessLoader` rooted at the document root. */
    static loader(driver, options) {
        return new SeleniumWebDriverHarnessEnvironment(() => driver.findElement(webdriver.By.css('body')), options);
    }
    /**
     * Flushes change detection and async tasks captured in the Angular zone.
     * In most cases it should not be necessary to call this manually. However, there may be some edge
     * cases where it is needed to fully flush animation events.
     */
    async forceStabilize() {
        await this.rawRootElement().getDriver().executeAsyncScript(whenStable);
    }
    /** @docs-private */
    async waitForTasksOutsideAngular() {
        // TODO: figure out how we can do this for the webdriver environment.
        //  https://github.com/angular/components/issues/17412
    }
    /** Gets the root element for the document. */
    getDocumentRoot() {
        return () => this.rawRootElement().getDriver().findElement(webdriver.By.css('body'));
    }
    /** Creates a `TestElement` from a raw element. */
    createTestElement(element) {
        return new SeleniumWebDriverElement(element, this._stabilizeCallback);
    }
    /** Creates a `HarnessLoader` rooted at the given raw element. */
    createEnvironment(element) {
        return new SeleniumWebDriverHarnessEnvironment(element, this._options);
    }
    // Note: This seems to be working, though we may need to re-evaluate if we encounter issues with
    // stale element references. `() => Promise<webdriver.WebElement[]>` seems like a more correct
    // return type, though supporting it would require changes to the public harness API.
    /**
     * Gets a list of all elements matching the given selector under this environment's root element.
     */
    async getAllRawElements(selector) {
        const els = await this._options.queryFn(selector, this.rawRootElement);
        return els.map((x) => () => x);
    }
}
//# sourceMappingURL=data:application/json;base64,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